<?php

namespace Database\Seeders;

use App\Enums\QuestionType;
use App\Models\Lecture;
use App\Models\LectureQuestion;
use App\Models\LectureQuestionCase;
use App\Models\PlacementTestTag;
use App\Models\QuestionOption;
use Illuminate\Database\Seeder;

class AllTypesQuestionsSeeder extends Seeder
{
    public $tag;
    public function __construct(){
        $this->tag=PlacementTestTag::all()->pluck('id')->toArray();
    }
    private array $readingPassages = [
        "The quick brown fox jumps over the lazy dog. This pangram contains all English letters.",
        "Solar energy is popular renewable source that reduces fossil fuel reliance.",
        "The Great Wall of China stretches over 13,000 miles for protection.",
        "Regular exercise improves health. 30 minutes daily is recommended.",
        "Gutenberg's printing press revolutionized book accessibility."
    ];

    private array $listeningPassages = [
        "The quick brown fox jumps over the lazy dog. This pangram contains all English letters.",
        "Solar energy is popular renewable source that reduces fossil fuel reliance.",
        "The Great Wall of China stretches over 13,000 miles for protection.",
        "Regular exercise improves health. 30 minutes daily is recommended.",
        "Gutenberg's printing press revolutionized book accessibility."
    ];

    private array $readingQuestions = [
        "What is this pangram's purpose?",
        "What's solar energy's advantage?",
        "How long is the Great Wall?",
        "What's the daily exercise recommendation?",
        "What did the printing press do?"
    ];

    private array $optionContents = [
        "For testing fonts and typing practice",
        "It reduces carbon emissions",
        "Over 13,000 miles",
        "At least 30 minutes of moderate activity",
        "It made books more accessible",
        "For teaching animals tricks",
        "It works at night",
        "About 1,000 miles",
        "One hour of intense exercise",
        "It increased the cost of books",
        "As a children's story",
        "It requires no maintenance",
        "Exactly 5,000 miles",
        "No exercise is needed",
        "It limited access to information",
        "For testing computer keyboards",
        "It only works in sunny countries",
        "Nearly 20,000 miles",
        "15 minutes of stretching",
        "It slowed down communication"
    ];

    private array $writingPrompts = [
        "Describe your favorite book and why you recommend it.",
        "Discuss the advantages and disadvantages of remote work.",
        "Write about a time you faced a challenge and how you overcame it.",
        "Explain the importance of environmental conservation.",
        "Compare and contrast traditional education with online learning."
    ];

    private array $caseStudyScenarios = [
        "A company is facing declining sales. Analyze potential causes and propose solutions.",
        "A patient is non-compliant with medical treatment. Develop a strategy to improve adherence.",
        "A school has high dropout rates. Identify factors and create an intervention plan.",
        "A tech startup needs to scale its operations. Outline a growth strategy.",
        "A community has poor waste management. Design a recycling initiative."
    ];

    public function run(): void
    {
        $lectures = Lecture::all();

        if ($lectures->isEmpty()) {
            $this->command->warn('No lectures found. Please seed lectures first.');
            return;
        }

        foreach ($lectures as $lecture) {

            for ($i = 0; $i < 10; $i++) {
                $this->createReadingQuestion($lecture);
            }

            for ($i = 0; $i < 8; $i++) {
                $this->createListeningQuestion($lecture);
            }

            for ($i = 0; $i < 5; $i++) {
                $this->createWritingQuestion($lecture);
            }

            for ($i = 0; $i < 5; $i++) {
                $this->createCaseQuestion($lecture);
            }
        }

        $this->command->info('Successfully seeded questions for all lectures.');
    }

    private function createReadingQuestion(Lecture $lecture): void
    {
        $index = rand(0, count($this->readingPassages) - 1);

        $question = LectureQuestion::create([
            'lecture_id' => $lecture->id,
            'tag_id' => $this->tag[rand(0,count($this->tag)-1)],
            'question_type' => QuestionType::Reading->value,
            'question_content' => "Passage:\n{$this->readingPassages[$index]}\n\nQuestion: {$this->readingQuestions[$index]}"
        ]);

        $this->createOptionsForQuestion($question, QuestionType::Reading->value);
    }

    private function createListeningQuestion(Lecture $lecture): void
    {
        $index = rand(0, count($this->listeningPassages) - 1);
        $question = LectureQuestion::create([
            'lecture_id' => $lecture->id,
            'tag_id' => $this->tag[rand(0,count($this->tag)-1)],
            'question_type' => QuestionType::Listening->value,
            'question_content' => "Passage:\n{$this->listeningPassages[$index]}\n\nQuestion: {$this->readingQuestions[$index]}",
            'listening_content' => 'lecture-questions/4766520250530021556mixkit-crickets-and-insects-in-the-wild-ambience-39.wav',
        ]);

        $this->createOptionsForQuestion($question, QuestionType::Listening->value);
    }

    private function createWritingQuestion(Lecture $lecture): void
    {
        $index = rand(0, count($this->writingPrompts) - 1);

        LectureQuestion::create([
            'lecture_id' => $lecture->id,
            'tag_id' => $this->tag[rand(0,count($this->tag)-1)],
            'question_type' => QuestionType::Writing->value,
            'question_content' => $this->writingPrompts[$index],
            'answer_writing'=>'answer'.$index,
        ]);
    }

    private function createCaseQuestion(Lecture $lecture): void
    {
        $index = rand(0, count($this->caseStudyScenarios) - 1);

        $case=LectureQuestionCase::create([
            'lecture_id' => $lecture->id,
            'case' => "Additional details for case study: " . $this->caseStudyScenarios[$index],
        ]);

        $question = LectureQuestion::create([
            'lecture_id' => $lecture->id,
            'tag_id' => $this->tag[rand(0,count($this->tag)-1)],
            'question_type' => QuestionType::Case->value,
            'question_content' => $this->caseStudyScenarios[$index],
            'case_id'=>$case->id,
        ]);

        $this->createOptionsForQuestion($question, QuestionType::Case->value);
    }

    private function createOptionsForQuestion(LectureQuestion $question, int $questionType): void
    {
        $optionsCount = 4;
        $correctOptionIndex = rand(0, $optionsCount - 1);

        for ($i = 0; $i < $optionsCount; $i++) {
            $isCorrect = $i === $correctOptionIndex;
            $optionContent = $this->generateOptionContent($questionType, $i);

            QuestionOption::create([
                'lecture_question_id' => $question->id,
                'option_content' => $optionContent,
                'is_correct' => $isCorrect
            ]);
        }
    }

    private function generateOptionContent(int $type, int $index): string
    {
        if ($type === QuestionType::Reading->value) {
            $start = ($index % 5) * 4;
            return $this->optionContents[$start + ($index % 4)] ?? 'Option ' . ($index + 1);
        }

        // For listening questions
        $start = (($index + 2) % 5) * 4;
        return $this->optionContents[$start + (($index + 1) % 4)] ?? 'Choice ' . ($index + 1);
    }
}
