<?php

namespace App\Services;

use Carbon\Carbon;
use App\Models\Lecture;
use App\Models\ZoomMeeting;
use Abdulbaset\ZoomIntegration\ZoomIntegrationService;

class ZoomMeetingService
{
    protected $zoom;

    public function __construct()
    {
        $this->zoom = new ZoomIntegrationService(config('services.zoom.account_id'), config('services.zoom.api_key'), config('services.zoom.api_secret'));
    }

    public function createMeeting(Lecture $lecture, array $data)
    {
        $meetingData  = $this->prepareMeetingData($lecture, $data);
        $zoomResponse = $this->zoom->createMeeting($meetingData);
        $lecture->update(['url' => $zoomResponse['response']['join_url']]);
        $this->createOrUpdateZoomMeetingData($lecture, $zoomResponse['response'], $data);
        return $zoomResponse;
    }

    // public function updateMeeting(Lecture $lecture, string $meetingId, array $data)
    // {
    //     $meetingData        = $this->prepareMeetingData($lecture, $data);
    //     $zoomResponse       = $this->zoom->updateMeeting($meetingId,$meetingData);
    //     $updatedMeetingData = $this->zoom->getMeeting($meetingId);
    //     $lecture->update(['link' => $updatedMeetingData['response']['join_url']]);
    //     $this->createOrUpdateZoomMeetingData($lecture, $updatedMeetingData['response'], $data);
    //     return $zoomResponse;
    // }

    public function updateMeeting(Lecture $lecture, string $meetingId, array $data)
    {
        $meetingData        = $this->prepareMeetingData($lecture, $data);
        $zoomResponse       = $this->zoom->updateMeeting($meetingId,$meetingData);
        $updatedMeetingData = $this->zoom->getMeeting($meetingId);
        if (!isset($updatedMeetingData['response']['id'])) {
            throw new \Exception("Failed to retrieve updated meeting details from Zoom.");
        }
        if (isset($updatedMeetingData['response']['join_url'])) {
            $lecture->update(['url' => $updatedMeetingData['response']['join_url']]);
        }
        $lecture->update(['url' => $updatedMeetingData['response']['join_url']]);
        $this->createOrUpdateZoomMeetingData($lecture, $updatedMeetingData['response'], $data);
        return $zoomResponse;
    }

    private function prepareMeetingData(Lecture $lecture, array $data): array
    {
        return [
            'topic'      => $lecture->name,
            'type'       => 2,
            'timezone'   => config('app.timezone'),
            'start_time' => Carbon::parse($data['meeting_start_time'])->toIso8601String(),
            'duration'   => $data['meeting_duration'],
            'settings'   => [
                'host_video'        => true,
                'participant_video' => true,
                'join_before_host'  => false,
                'mute_upon_entry'   => true,
                'waiting_room'      => true
            ]
        ];
    }

    private function createOrUpdateZoomMeetingData(Lecture $lecture, array $zoomResponse, array $data): void
    {
        ZoomMeeting::updateOrCreate(
            ['lecture_id' => $lecture->id],
            [
                'ref_id'     => $zoomResponse['id'],
                'topic'      => $zoomResponse['topic'],
                'start_time' => Carbon::parse($data['meeting_start_time'])->format('Y-m-d H:i:s'),
                'duration'   => $data['meeting_duration'],
                'join_url'   => $zoomResponse['join_url'],
                'start_url'  => $zoomResponse['start_url'],
                'agenda'     => $data['agenda'] ?? null,
            ]
        );
    }

    public function getMeetingRecordings(Lecture $lecture)
    {
        if (!$lecture->zoomMeetings || !$lecture->zoomMeetings->ref_id) {
            throw new \Exception("No Zoom meeting associated with this lecture.");
        }

        return $this->zoom->getMeetingRecordings($lecture->zoomMeetings->ref_id);
    }

}
