<?php

namespace App\Services;

use App\Enums\DefineStatus;
use App\Enums\UserType;
use App\Http\Resources\User\UserResource;
use App\Http\Traits\ApiResponse;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;

class UserAuthService
{
    use ApiResponse;
    
    /**
     * Authenticate user and generate an access token.
     *
     * @param string $email The user's email address.
     * @param string $password The user's password.
     * @param UserType $type The type of user attempting to log in.
     * 
     * @return JsonResponse The authentication response with a token.
     */
    public function login(string $email, string $password, UserType $type): JsonResponse
    {
        $user = match ($type) {
            UserType::ADMIN => $this->getAdminUser($email),
            UserType::STUDENT => $this->getStudentUser($email),
            UserType::INSTRUCTOR => $this->getInstructorUser($email),
            UserType::MENTOR => $this->getSupervisorUser($email),
            default => null,
        };

        if (!$user || !Hash::check($password, $user->password)) {
            return $this->apiResponse([], __('main.incorrect_credentials'), 401);
        }

        return $this->ensureUserIsActive($user) ?? $this->generateTokenResponse($user, $type);
    }

    /**
     * Ensure the user is active before allowing login.
     *
     * @param User $user The user to check.
     * @return JsonResponse|null Returns error response if inactive, otherwise null.
     */
    private function ensureUserIsActive(User $user): ?JsonResponse
    {
        if ($user->status !== DefineStatus::ACTIVE) {
            return $this->apiResponse([], __('main.account_inactive'), 403);
        }
        return null;
    }

    /**
     * Generate API token for authenticated user.
     * @param User $user The authenticated user.
     * @param UserType $type The type of the authenticated user.
     * @return JsonResponse The response containing the API token.
     */
    private function generateTokenResponse(User $user, UserType $type): JsonResponse
    {
        $token = $user->createToken('UserToken', ["type:" . strtolower($type->name)])->plainTextToken;

        return $this->apiResponse([
            'token' => $token,
            'user' => new UserResource($user),
        ], __('main.logged_in'), 200);
    }

    /**
     * Retrieve an admin user by email.
     * @param string $email The email of the admin user.
     * @return User|null The admin user or null if not found.
    */
    private function getAdminUser(string $email): ?User
    {
        return User::admins()->where('email', $email)->first();
    }

    /**
     * Retrieve a student user by email.
     * @param string $email The email of the student user.
     * @return User|null The student user or null if not found.
    */
    private function getStudentUser(string $email): ?User
    {
        return User::students()->where('email', $email)->first();
    }

    /**
     * Retrieve an instructor user by email.
     * @param string $email The email of the instructor user.
     * @return User|null The instructor user or null if not found.
     */
    private function getInstructorUser(string $email): ?User
    {
        return User::instructors()->where('email', $email)->first();
    }

    /**
     * Retrieve a supervisor user by email.
     * @param string $email The email of the supervisor user.
     * @return User|null The supervisor user or null if not found.
     */
    private function getSupervisorUser(string $email): ?User
    {
        return User::supervisors()->where('email', $email)->first();
    }

    /**
     * Revoke the user's current access token (logout).
     * @param User $user The user who is logging out.
     * @return JsonResponse The logout confirmation response.
     */
    public function logout(User $user): JsonResponse
    {
        $user->currentAccessToken()->delete();
        return $this->apiResponse([], __('main.logged_out'), 200);
    }
}
