<?php

namespace App\Services;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Round;
use App\Models\Course;
use App\Models\Payment;
use App\Models\CourseEnrollment;
use Illuminate\Support\Collection;
use Illuminate\Database\Eloquent\Builder;

class ReportService
{
    public function getCounts()
    {
        $totalAmountsPaid = CourseEnrollment::where('payment_status', 1)->sum('total_amount');
        $studentsCount    = User::students()->count();
        $coursesCount     = Course::count();
        $groupsCount      = Round::count();
        return [
            'total_amounts_paid' => (float)$totalAmountsPaid,
            'students_count'     => (int)$studentsCount,
            'courses_count'      => (int)$coursesCount,
            'groups_count'       => (int)$groupsCount,
        ];
    }

    /**
     * Generate a report with data grouped by months and filtered by specified conditions.
     *
     * @param Builder $query Query builder instance to fetch data.
     * @param array $filters Filters to apply on the query.
     * @param string $dateColumn The column used for date-based filtering (default: 'created_at').
     * @return array The report data including monthly data, total count, and available years.
     */
    public function generateReport(Builder $query, array $filters, string $dateColumn = 'created_at'): array
    {
        $year        = $filters['year'] ?? Carbon::now()->year;
        $startOfYear = Carbon::create($year, 1, 1)->startOfYear();
        $endOfYear   = Carbon::create($year, 12, 31)->endOfYear();
        // Apply filters to the query
        $query = $this->applyFilters($query, $filters, $dateColumn);
        // Get data grouped by month
        $yearData = clone $query;
        $yearData = $yearData
            ->whereBetween($dateColumn, [$startOfYear, $endOfYear])
            ->get()
            ->groupBy(function ($item) use ($dateColumn) {
                return Carbon::parse($item->$dateColumn)->month;
            });
        // Prepare monthly data
        $monthlyData = $this->generateMonthlyData($yearData);
        return [
            'total_count'         => $query->count(),
            'current_month_count' => $monthlyData[Carbon::now()->month] ?? 0,
            'monthly_data'        => $monthlyData,
            'available_years'     => $this->getYearRange($query, $dateColumn),
        ];
    }

    /**
     * Apply filters to the query based on the provided filter criteria.
     *
     * @param Builder $query Query builder instance.
     * @param array $filters Filters to apply on the query.
     * @param string $dateColumn The column used for date-based filtering.
     * @return Builder The modified query builder instance.
     */
    private function applyFilters(Builder $query, array $filters, string $dateColumn): Builder
    {
        return $query
            ->when(isset($filters['course_id']), fn($q) => $q->where('course_id', $filters['course_id']))
            ->when(isset($filters['status']), fn($q) => $q->where('status', $filters['status']))
            ->when(isset($filters['from_date']), fn($q) => $q->whereDate($dateColumn, '>=', $filters['from_date']))
            ->when(isset($filters['to_date']), fn($q) => $q->whereDate($dateColumn, '<=', $filters['to_date']));
    }

    /**
     * Generate monthly data based on grouped data.
     *
     * @param Collection $data Grouped data by month.
     * @return array An array with counts for each month.
     */
    private function generateMonthlyData(Collection $data): array
    {
        $monthlyData = [];
        for ($month = 1; $month <= 12; $month++) {
            $monthName = Carbon::create()->month($month)->format('M');
            $monthlyData[$monthName] = 0;
        }


        foreach ($data as $month => $items) {
            $monthName = Carbon::create()->month($month)->format('M');
            $monthlyData[$monthName] = $items->count();
        }

        return $monthlyData;
    }

    /**
     * Determine the range of years available in the dataset based on the date column.
     *
     * @param Builder $query Query builder instance.
     * @param string $dateColumn The column used for date-based filtering.
     * @return array An array of years from the earliest record to the current year.
     */
    private function getYearRange(Builder $query, string $dateColumn): array
    {
        $firstRecord = (clone $query)->oldest($dateColumn)->first();
        $startYear = $firstRecord ? Carbon::parse($firstRecord->$dateColumn)->year : Carbon::now()->year;
        $endYear = Carbon::now()->year;

        return range($startYear, $endYear);
    }
}
