<?php

namespace App\Services\Payment;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use App\Interfaces\PaymentGatewayInterface;
use App\Services\Payment\BasePaymentService;


class PaymobPaymentService extends BasePaymentService implements PaymentGatewayInterface
{
    protected string $secretKey;
    protected array $integrationIds;
    protected string $baseUrl;
    protected array $headers;
    protected string $currency;
      /**
     * Constructor to initialize PaymobPaymentService.
     */
    public function __construct()
    {
        $this->baseUrl        = env('PAYMOB_BASE_URL');
        $this->secretKey      = env('PAYMOB_API_SECRET_KEY');
        $this->integrationIds = explode(',', env('PAYMOB_INTEGRATION_IDS'));
        $this->currency       = "EGP";
        $this->headers        = [
            'Accept'        => 'application/json',
            'Content-Type'  => 'application/json',
            'Authorization' => 'Token ' . $this->secretKey,
        ];
    }


    public function sendPayment(array $paymentData): array
    {
        try {
            $url                            = '/v1/intention/';
            $paymentData['currency']        = $this->currency;
            $paymentData['payment_methods'] = $this->integrationIds;
            
            $response     = $this->buildRequest('POST', $this->baseUrl.$url,$this->headers, $paymentData, 'json');
            $responseData = $response->getData();
            if (empty($responseData->data)) {
                return $this->response(false, 'Invalid response from payment gateway');
            }
            $clientSecret = $responseData->data->client_secret ?? null;
            if (!$clientSecret) {
                Log::error('Payment error ' . json_encode($responseData));
                if (isset($responseData->data->billing_data)) {
                    $errors = [];
                    foreach ($responseData->data->billing_data as $field => $messages) {
                        if (is_array($messages)) {
                            $errors[] = $messages[0];
                        }
                    }
                    return $this->response(false, implode(', ', $errors));
                }
                if (isset($responseData->data->detail)) {
                    return $this->response(false, str_replace('_', ' ', $responseData->data->detail));
                }
                return $this->response(false, 'Client secret missing in response');
            }
            $redirectUrl = $this->generateRedirectUrl($clientSecret);
            return $this->response(true, 'Payment initiated successfully', [
                'redirect_url' => $redirectUrl,
                'payment_id'   => $responseData->data->id ?? null,
                'data'         => $responseData->data,
            ]);
        } catch (\Exception $e) {
            Log::error('Payment initiation failed: ' . $e->getMessage());
            return $this->response(false, 'Payment initiation failed: ' . $e->getMessage());
        }
    }


    private function generateRedirectUrl(string $clientSecret): string
    {
        $publicKey = config('services.paymob.public_key');
        return "https://accept.paymob.com/unifiedcheckout/?publicKey={$publicKey}&clientSecret={$clientSecret}";
    }


    private function response(bool $success, string $message, array $data = null): array
    {
        return [
            'success' => $success,
            'message' => $message,
            'data'    => $data,
        ];
    }

    public function callBack(Request $request): bool
    {
        $response = $request->all();
        Storage::put('paymob_response.json', json_encode($response));
        return isset($response['success']) && $response['success'] === 'true';
    }


}
