<?php
namespace App\Services;

use Carbon\Carbon;
use App\Services\WaSenderService;
use Illuminate\Support\Facades\DB;

class OtpService
{
    protected $table = 'otps';

    public function __construct(protected WaSenderService $waSenderService)
    {
        // Initialize OTP service
    }

    /**
     * Generate a numeric OTP code
     *
     * @param string $identifier User identifier
     * @param int $length Length of OTP code
     * @param int $validity Validity time in minutes
     * @return object Response with token and status
     */
    public function generateOtp($identifier, $length = 4, $validity = 10)
    {
        // Delete any existing valid OTPs for this identifier
        $this->deleteOtp($identifier);

        // Generate OTP code
        $token = $this->generateNumericToken($length);

        // Store OTP in database
        $this->storeOtp($identifier, $token, $validity);
        //add +20 to the phone number
        $EgyptionIdentifier = "+2" . $identifier;
        // dd($EgyptionIdentifier);
        // if($EgyptionIdentifier != "+201155911826"){
        //     $this->waSenderService->sendMessage($EgyptionIdentifier, "Hello, your OTP code is: " . $token.".\n\nThank you for using EngliVision.");
        // }
        $this->waSenderService->sendMessage($EgyptionIdentifier, "Hello, your OTP code is: " . $token.".\n\nThank you for using EngliVision.");
        return (object)[
            'status' => true,
            'token' => $token,
            'message' => __('main.otp_generated_successfully')
        ];
    }

    /**
     * Generate a numeric token of specified length
     *
     * @param int $length Length of token
     * @return string Generated token
     */
    private function generateNumericToken($length = 4)
    {
        $token = "";
        for ($i = 0; $i < $length; $i++) {
            $token .= mt_rand(0, 9);
        }
        return $token;
    }

    /**
     * Store OTP in database
     *
     * @param string $identifier User identifier
     * @param string $otp OTP code
     * @param int $validity Validity time in minutes
     * @return void
     */
    public function storeOtp($identifier, $otp, $validity = 10)
    {
        DB::table($this->table)->insert([
            'identifier'    => $identifier,
            'token'          => $otp,
            'validity_time' => Carbon::now()->addMinutes($validity),
            'is_verified'   => false,
            'created_at'    => Carbon::now(),
            'updated_at'    => Carbon::now()
        ]);
    }

    /**
     * Send OTP to user (placeholder for actual implementation)
     *
     * @param string $otp OTP code
     * @param string $activity Purpose of OTP
     * @param int|null $expiryTime Expiry time in minutes
     * @return bool Success status
     */
    public function sendOtp($otp, $activity = 'register', $expiryTime = null)
    {
        // Implementation for sending OTP via SMS or email would go here
        return true;
    }

    /**
     * Get the most recent OTP record for an identifier
     *
     * @param string $identifier User identifier
     * @return object|null OTP record
     */
    public function getOtpRecord($identifier)
    {
        return DB::table($this->table)
            ->where('identifier', $identifier)
            ->orderByDesc('created_at')
            ->first();
    }

    /**
     * Delete all OTPs for an identifier
     *
     * @param string $identifier User identifier
     * @return void
     */
    public function deleteOtp($identifier)
    {
        DB::table($this->table)
            ->where('identifier', $identifier)
            ->delete();
    }

    /**
     * Mark an OTP as verified
     *
     * @param string $identifier User identifier
     * @return void
     */
    public function markAsVerified($identifier)
    {
        DB::table($this->table)
            ->where('identifier', $identifier)
            ->update([
                'is_verified' => true,
                'updated_at' => Carbon::now()
            ]);
    }

    /**
     * Verify an OTP code
     *
     * @param string $identifier User identifier
     * @param string $otp OTP code to verify
     * @return object Response with status and message
     */
    public function verifyOtp($identifier, $otp, $markAsVerified = true)
    {
        $otpRecord = $this->getOtpRecord($identifier);

        if (!$otpRecord) {
            return (object)[
                'status' => false,
                'message' => __('main.otp.otp_does_not_exist')
            ];
        }

        if ($otpRecord->is_verified && $markAsVerified) {
            return (object)[
                'status' => false,
                'message' => __('main.otp.otp_already_used')
            ];
        }
        if ($otpRecord->token != $otp) {
            return (object)[
                'status' => false,
                'message' => __('main.otp.invalid_otp')
            ];
        }

        $isOtpValid = Carbon::now()->lessThanOrEqualTo(Carbon::parse($otpRecord->validity_time));
        if (!$isOtpValid) {
            return (object)[
                'status' => false,
                'message' => __('main.otp.otp_expired')
            ];
        }

        if ($markAsVerified) {
            $this->markAsVerified($identifier);
        }

        return (object)[
            'status' => true,
            'message' => __('main.otp.otp_is_valid')
        ];
    }
}
