<?php
namespace App\Services;

use App\Models\User;
use App\Http\Traits\ApiResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;
use App\Http\Resources\User\UserResource;
use Illuminate\Validation\ValidationException;

class AuthService
{
    use ApiResponse;

    public function login(array $credentials, string $userType): JsonResponse
    {
        // Find the user by email
        $user = User::where('email', $credentials['email']);

        // Apply scope or condition based on user type
        switch ($userType) {
            case 'admin':
                $user->admins();
                break;
            case 'student':
                $user->students();
                break;
            case 'supervisor':
                $user->supervisors();
                break;
            case 'instructor':
                $user->instructors();
                break;
            default:
                throw ValidationException::withMessages(['email' => __('main.invalid_user_type')]);
        }

        $user = $user->first();

        // Validate user credentials
        if (!$user || !Hash::check($credentials['password'], $user->password)) {
            return $this->apiResponse([], __('main.incorrect_credentials'), 401);
        }

        // Generate Token with user-specific abilities
        $token = $user->createToken('UserToken', ["type:$userType"])->plainTextToken;

        return $this->apiResponse([
            'token' => $token,
            'user' => new UserResource($user->load('role.permissions')),
        ], __('main.logged_in'), 200);
    }
    public function logout(): JsonResponse
    {
        auth()->user()->currentAccessToken()->delete();
        return $this->apiResponse([],__('main.logged_out'), 200);
    }
}
