<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\HtmlString;

class InvoicePaidMail extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * The invoice data.
     *
     * @var array
     */
    protected $data;

    /**
     * The invoice file path.
     *
     * @var string
     */
    protected $invoicePath;

    /**
     * Create a new notification instance.
     *
     * @param  string  $invoicePath
     * @param  array  $data
     * @return void
     */
    public function __construct(string $invoicePath, array $data = [])
    {
        $this->invoicePath = $invoicePath;
        $this->data = array_merge([
            'title' => 'Invoice Payment Confirmation',
            'message' => 'Thank you for your payment. Your invoice is attached.',
            'subject' => 'Invoice Paid Successfully',
            'action_url' => null,
            'action_text' => 'View Invoice',
        ], $data);
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable): array
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable): MailMessage
    {
        $mailMessage = (new MailMessage)
            ->subject($this->data['subject'])
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line(new HtmlString('<strong>' . $this->data['title'] . '</strong>'))
            ->line($this->data['message'])
            ->attach(storage_path('app/public/' . $this->invoicePath), [
                'as' => 'invoice_' . basename($this->invoicePath),
                'mime' => 'application/pdf',
            ]);

        $mailMessage->line('Thank you for using our application!')
            ->salutation(new HtmlString('Regards,<br>' . config('app.name')));

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable): array
    {
        return [
            'invoice_path' => $this->invoicePath,
            'data' => $this->data,
        ];
    }
}
