<?php

namespace App\Models;

use Essa\APIToolKit\Filters\Filterable;
use Illuminate\Database\Eloquent\Model;
use App\Enums\TransactionTypeEnum;
use App\Filters\TransactionFilters;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class Transaction extends Model
{
     use Filterable;
    protected string $default_filters = TransactionFilters::class;
    protected $fillable = [
        'user_id', 'course_enrollment_id', 'installment_id',
        'amount', 'payment_method', 'payment_date',
        'type', 'notes', 'accepted_by', 'created_by', 'updated_by','payment_method_id','reference_code'
    ];


    protected $casts = [
        'payment_date' => 'date',
        'type' => TransactionTypeEnum::class,
    ];

    public static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (Auth::check()) {
                $model->created_by = $model->created_by ?? Auth::id();
            }
            PaymentMethod::where('id', $model->payment_method_id)
                ->increment('total_amount', $model->amount);
        });

        static::updating(function ($model) {
            if (Auth::check()) {
                $model->updated_by = Auth::id();
            }
        });

        static::created(function ($model) {
            $date = Carbon::now()->format('Y-m-d');
            $hour = Carbon::now()->format('H');
            $model->reference_code = "{$date}{$model->user_id}{$hour}";
            $model->saveQuietly();
        });
    }

    public function paymentMethod()
    {
        return $this->belongsTo(PaymentMethod::class, 'payment_method_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class,'user_id');
    }

    public function courseEnrollment()
    {
        return $this->belongsTo(CourseEnrollment::class, 'course_enrollment_id');
    }

    public function installment()
    {
        return $this->belongsTo(Installment::class,'installment_id');
    }

    public function acceptedBy()
    {
        return $this->belongsTo(User::class, 'accepted_by');
    }

    public function getAcceptedByUserAttribute()
    {
        return $this->relationLoaded('acceptedBy')
            ? $this->acceptedBy?->name
            : null;
    }

}

