<?php

namespace App\Models;

use App\Enums\SurveyQuestionType;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SurveyAnswer extends Model
{
    protected $fillable = [
        'survey_id',
        'survey_section_id',
        'survey_question_id',
        'user_id',
        'answer',
        'question_type',
        'request_ip',
        'created_at',
        'updated_at',
    ];

    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'answer' => 'array',
    ];

    public static function boot(){
        parent::boot();
        static::creating(function (SurveyAnswer $surveyAnswer) {
            $surveyAnswer->request_ip = request()->ip();
            if (Auth::guard('sanctum')->check()) {
                $surveyAnswer->user_id = Auth::guard('sanctum')->user()->id;
            }
        });
    }

    /**
     * Get the user that owns the survey answer.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function setAnswerAttribute($value)
    {
        if (is_array($value)) {
            $this->attributes['answer'] = json_encode($value, JSON_UNESCAPED_UNICODE);
        } else if (is_string($value)) {
            $decoded = json_decode($value, true);
            $this->attributes['answer'] = $decoded !== null ? json_encode($decoded, JSON_UNESCAPED_UNICODE) : $value;
        } else {
            $this->attributes['answer'] = $value;
        }
    }

    public function getAnswerAttribute($value)
    {
        if (is_null($value)) {
            return null;
        }

        if (is_array($value)) {
            return $value;
        }

        if (is_string($value) && (str_starts_with($value, '{') || str_starts_with($value, '['))) {
            $decoded = json_decode($value, true);
            return json_last_error() === JSON_ERROR_NONE ? $decoded : $value;
        }

        return $value;
    }

    public function survey(): BelongsTo
    {
        return $this->belongsTo(Survey::class);
    }

    public function section(): BelongsTo
    {
        return $this->belongsTo(SurveySection::class, 'survey_section_id');
    }

    public function question(): BelongsTo
    {
        return $this->belongsTo(SurveyQuestion::class, 'survey_question_id');
    }
}
