<?php

namespace App\Models;

use App\Enums\DefineStatus;
use App\Enums\TypeOfRound;
use App\Filters\RoundFilters;
use Essa\APIToolKit\Filters\Filterable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Facades\DB;


class Round extends Model
{
    use HasFactory, Filterable;
    protected string $default_filters = RoundFilters::class;
    protected $casts = [
        'status' => DefineStatus::class,
        'type' => TypeOfRound::class,
        'additional_instructors'=>'array',
        'round_content'=>'array',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
    ];

    protected $fillable = [
        'course_id',
        'instructor_id',
        'name',
        'start_date',
        'end_date',
        'discount',
        'students_count',
        'whatsapp_group_link',
        'status',
        'google_meet_link',
        'type',
        'branch_id',
        'additional_instructors',
        'round_content',
        'max_student',
        'link_url',
        'weeks_count',
        'days_count',
        'day_one',
        'day_two',
    ];

    protected static function booted()
    {
        static::creating(function ($round) {
            if ($round->course_id) {
                $round->round_content = Course::query()
                    ->whereKey($round->course_id)
                    ->value('course_content');
            }
        });
        static::created(function ($round) {
            self::generateLectures($round);
        });
    }

    public static function generateLectures(Round $round)
    {
        $now = now();
        $lectures = [];
        $lectureTasks = [];
        $startDate = \Carbon\Carbon::parse($round->start_date);

        $dayMap = [
            'sunday' => 'sunday',
            'monday' => 'monday',
            'tuesday' => 'tuesday',
            'wednesday' => 'wednesday',
            'thursday' => 'thursday',
            'friday' => 'friday',
            'saturday' => 'saturday',
        ];

        if (empty($round->day_one) || !isset($dayMap[strtolower($round->day_one)])) {
            throw new \Exception('Please specify at least one valid day for the lectures');
        }

        $firstDay = $dayMap[strtolower($round->day_one)];
        $hasSecondDay = !empty($round->day_two) && isset($dayMap[strtolower($round->day_two)]);
        $secondDay = $hasSecondDay ? $dayMap[strtolower($round->day_two)] : null;

        if ($hasSecondDay && $secondDay === $firstDay) {
            $hasSecondDay = false;
        }

        for ($week = 1; $week <= $round->weeks_count; $week++) {
            $firstDayDate = (clone $startDate)->addWeeks($week - 1)->next($firstDay);

            if ($firstDayDate->lt($startDate)) {
                $firstDayDate->addWeek();
            }
            $secondDayDate = null;
            if ($hasSecondDay) {
                $secondDayDate = (clone $startDate)->addWeeks($week - 1)->next($secondDay);
                if ($secondDayDate->lt($startDate)) {
                    $secondDayDate->addWeek();
                }
            }

            $lectures[] = [
                'round_id' => $round->id,
                'name' => "Week {$week} - " . ucfirst($firstDay),
                'url' => null,
                'start_time' => $firstDayDate->toDateTimeString(),
                'week_number' => $week,
                'status' => DefineStatus::ACTIVE,
                'created_at' => $now,
                'updated_at' => $now,
            ];

            if ($hasSecondDay) {
                $lectures[] = [
                    'round_id' => $round->id,
                    'name' => "Week {$week} - " . ucfirst($secondDay),
                    'url' => null,
                    'start_time' => $secondDayDate->toDateTimeString(),
                    'week_number' => $week,
                    'status' => DefineStatus::ACTIVE,
                    'created_at' => $now,
                    'updated_at' => $now,
                ];
            }
        }

        Lecture::insert($lectures);

        $insertedLectures = Lecture::where('round_id', $round->id)->get();

        foreach ($insertedLectures as $lecture) {
            $startTime = \Carbon\Carbon::parse($lecture->start_time);

            $lectureTasks[] = [
                'lecture_id' => $lecture->id,
                'description' => 'lecture task',
                'goal' => 'goal',
                'note' => null,
                'files_included' => null,
                'deadline' => $startTime->copy()->addDays(5),
                'task_link' => null,
                'created_at' => $now,
                'updated_at' => $now,
            ];
        }

        LectureTask::insert($lectureTasks);
    }


    public function notices(){
        return $this->hasMany(Notice::class,'round_id','id');
    }
    public function studentRounds(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'round_students')
                    ->withPivot('status', 'transaction_id', 'total_amount', 'enrollment_date', 'payment_method', 'payment_status')
                    ->withTimestamps();
    }

    public function hasEnded(): bool
    {
        return \Carbon\Carbon::parse($this->end_date)->endOfDay()->isPast();
    }

    public function hasNotStarted(): bool
    {
        return \Carbon\Carbon::parse($this->start_date)->startOfDay()->isFuture();
    }

    public function isRunning(): bool
    {
        return now()->between(
            \Carbon\Carbon::parse($this->start_date)->startOfDay(),
            \Carbon\Carbon::parse($this->end_date)->endOfDay()
        );
    }

    public function course(): BelongsTo
    {
        return $this->belongsTo(Course::class);
    }

    public function lectures(): HasMany
    {
        return $this->hasMany(Lecture::class);
    }

    public function instructor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'instructor_id');
    }


    public function supervisors(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'round_supervisor')
                    ->withTimestamps();
    }

    public function students(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'round_students')
                    ->withTimestamps();
    }



    public function getUserPoints(User $user)
    {
        return RoundStudent::where('user_id', $user->id)->where('round_id', $this->id)->first()->total_points;
    }

    public function branch()
    {
        return $this->belongsTo(Branche::class);
    }

    public function courseEnrollments()
    {
        return $this->hasMany(CourseEnrollment::class,'round_id','id');
    }

    public function myCourseEnrollment()
    {
        return $this->hasOne(CourseEnrollment::class, 'round_id')
                    ->where('user_id', auth()->id());
    }


    public function transactions()
    {
        return $this->hasManyThrough(
            Transaction::class,
            CourseEnrollment::class,
            'round_id',
            'course_enrollment_id',
            'id',
            'id'
        );
    }

    public function installments()
    {
        return $this->hasManyThrough(
            Installment::class,
            CourseEnrollment::class,
            'round_id',
            'course_enrollment_id',
            'id',
            'id'
        );
    }


    public function getAbsence($roundId)
    {
        $userId=auth()->id();
        $round=Round::find($roundId);
        $lectureIds = $round->lectures()->pluck('lectures.id');

        $totalLectures = $lectureIds->count();
        if ($totalLectures === 0) return 0;

        $actualAttendances = Evaluation::whereIn('lecture_id', $lectureIds)
            ->where('user_id', $userId)
            ->where('is_attend', false)
            ->count();
        return [
            'absence_rate'=>($actualAttendances / $totalLectures) * 100,
            'absence_count'=>$actualAttendances
        ];
    }

    public function taskSubmissionsDegreeSummary()
    {
        $lectureIds = $this->lectures()->pluck('lectures.id');
        $studentIds = $this->students()->pluck('users.id');

        $totalCount=$lectureIds->count() * $studentIds->count();

        if ($lectureIds->isEmpty() || $studentIds->isEmpty()) {
            return '0/0';
        }

        $query = TaskSubmission::whereIn('user_id', $studentIds)
            ->whereIn('lecture_id', $lectureIds);

        $correctedCount = (clone $query)->where('task_degree', '>=', 0)->count();

        return $correctedCount . '/' . $totalCount;
    }







    public function getAttendanceRate()
    {
        $lectureIds = $this->lectures()->pluck('lectures.id');
        $studentIds = $this->students()->pluck('users.id');

        $totalPossibleAttendances = $lectureIds->count() * $studentIds->count();

        if ($totalPossibleAttendances === 0) return 0;

        $actualAttendances = Evaluation::whereIn('lecture_id', $lectureIds)
            ->whereIn('user_id', $studentIds)
            ->where('is_attend', true)
            ->count();

        return round(($actualAttendances / $totalPossibleAttendances) * 100, 2);
    }

    public function getAttendanceRateByUser()
    {
        $lectureIds = $this->lectures()->pluck('lectures.id');
        $totalLectures = $lectureIds->count();

        if ($totalLectures === 0) return 0;

        $actualAttendances = Evaluation::whereIn('lecture_id', $lectureIds)
            ->where('user_id', auth()->id())
            ->where('is_attend', true)
            ->count();

        return round(($actualAttendances / $totalLectures) * 100, 2);
    }

    public function getTaskSubmissionRate()
    {
        $lectureIds = $this->lectures()->pluck('lectures.id');
        $studentIds = $this->students()->pluck('users.id');

        $taskSubmissions = TaskSubmission::whereIn('user_id', $studentIds)
            ->whereIn('lecture_id', $lectureIds)
            ->get();

        $totalEvaluations = $taskSubmissions->sum('task_degree');
        $maxPossibleScore = $lectureIds->count() * $studentIds->count() * 10;

        if ($maxPossibleScore === 0) return 0;

        return round(($totalEvaluations / $maxPossibleScore) * 100, 2);
    }

    public function getTaskSubmissionRateByUser()
    {
        $lectureIds = $this->lectures()
            ->whereHas('taskSubmissions')
            ->pluck('lectures.id');

        $lectureCount = $lectureIds->count();
        $maxPossibleScore = $lectureCount * 10;

        if ($maxPossibleScore === 0) return 0;

        $taskSubmissions = TaskSubmission::where('user_id', auth()->id())
            ->whereIn('lecture_id', $lectureIds)
            ->get();

        $totalEvaluations = $taskSubmissions->sum('task_degree');

        return round(($totalEvaluations / $maxPossibleScore) * 100, 2);
    }


    // public function getAverageRating(): ?float
    // {
    //     $lectureIds = $this->lectures()->pluck('lectures.id');

    //     if ($lectureIds->isEmpty()) {
    //         return null;
    //     }

    //     $evaluations = Evaluation::whereIn('lecture_id', $lectureIds)->get();

    //     if ($evaluations->isEmpty()) {
    //         return null;
    //     }

    //     $totalEvaluations = $evaluations->count();
    //     $attendedCount = $evaluations->where('is_attend', true)->count();

    //     return round(($attendedCount / $totalEvaluations) * 100, 2);
    // }

}
