<?php

namespace App\Imports;

use App\Models\Lecture;
use App\Enums\DefineStatus;
use App\Models\Round;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;

class ImportLecture implements ToModel, WithHeadingRow, WithValidation
{
    private array $errors = [];

    public function model(array $row)
    {
        try {
            $roundId = $this->getRoundId($row);

            if (!$roundId) {
                $this->errors[] = "Round not found for group: " . ($row['group_name'] ?? 'N/A');
                return null;
            }

            $numberOfLectures = (int)($row['number_lectures'] ?? 0);

            if ($numberOfLectures <= 0) {
                $this->errors[] = "Invalid number of lectures for group: " . $row['group_name'];
                return null;
            }

            // Lecture::where('round_id', $roundId)->delete();

            for ($i = 1; $i <= $numberOfLectures; $i++) {
                $lectureName = 'Lecture ' . $i;
                Lecture::updateOrCreate(
                    [
                    'round_id' => $roundId,
                        'name' => $lectureName,
                    ],
                    [
                        'week_number' => $i,
                        'url' => $row['url'] ?? null,
                        'start_time' => now()->addWeeks($i),
                        'status' => DefineStatus::ACTIVE,
                    ]
                );
            }

            return null;
        } catch (\Exception $e) {
            Log::error('Error importing lecture: ' . $e->getMessage(), [
                'row' => $row,
                'trace' => $e->getTraceAsString()
            ]);
            $this->errors[] = 'Error processing row: ' . $e->getMessage();
            return null;
        }
    }

    private function getRoundId(array $row): ?int
    {
        $groupName = $row['group_name'] ?? null;

        if (!$groupName) {
            return null;
        }

        $round=Round::where('name', $groupName)->first();
        if(!$round){
            return null;
        }
        return $round->id;
    }

    public function rules(): array
    {
        return [
            'group_name' => [
                'required',
                'string',
                'max:255',
            ],
            'number_lectures' => 'required|integer|min:1|max:52', // Assuming max 52 weeks in a year
            'url' => 'nullable|url|max:1000',
        ];
    }

    public function customValidationMessages()
    {
        return [
            'group_name.exists' => 'The selected group does not exist or is not active.',
            'number_lectures.max' => 'Cannot create more than 52 lectures at once.',
        ];
    }

    public function getErrors(): array
    {
        return $this->errors;
    }
}
