<?php

namespace App\Http\Middleware;

use App\Models\ActivityLog;
use Closure;
use Illuminate\Http\Request;

class LogAdminActivity
{
    public function handle(Request $request, Closure $next)
    {
        $response = $next($request);

        $user = $request->user();
        $routeName = $request->route()?->getName();

        $action = $this->getActionFromRequest($request);
        $message = $this->getMessageFromAction($action, $request);

        if ($user) {
            ActivityLog::log(
                action: $action,
                message: $message,
                context: [
                    'route' => $routeName,
                    'method' => $request->method(),
                    'url' => $request->fullUrl(),
                    'data' => $this->getFilteredRequestData($request)
                ]
            );
        }

        return $response;
    }

    private function getActionFromRequest(Request $request): string
    {
        $method = $request->method();

        return match(true) {
            $method === 'POST' && $request->path() === 'login' => 'login',
            $request->path() === 'logout' => 'logout',
            $method === 'POST' => 'create',
            $method === 'PUT' || $method === 'PATCH' => 'update',
            $method === 'DELETE' => 'delete',
            default => 'view',
        };
    }

    private function getMessageFromAction(string $action, Request $request): string
    {
        $messages = [
            'create' => 'Created new record',
            'update' => 'Updated record',
            'delete' => 'Deleted record',
            'login' => 'User logged in',
            'logout' => 'User logged out',
            'view' => 'Viewed page',
        ];

        return $messages[$action] ?? 'Performed action';
    }

    private function getFilteredRequestData(Request $request): array
    {
        $data = $request->except(['_token', '_method', 'password', 'password_confirmation']);

        return $data;
    }
}
