<?php

namespace App\Http\Controllers\Api\V1;

use App\Models\User;
use App\Enums\UserType;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use App\Http\Resources\User\UserResource;
use App\Http\Requests\User\CreateUserRequest;
use App\Http\Requests\User\UpdateUserRequest;
use App\Http\Traits\MediaHandler;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;


class SupervisorController extends Controller
{
    use MediaHandler;

    public function index(): AnonymousResourceCollection
    {
        $supervisors = User::Supervisors()
            ->useFilters() 
            ->latest()
            ->dynamicPaginate();
        return UserResource::collection($supervisors);
    }

    public function store(CreateUserRequest $request): JsonResponse
    {
        $data = $request->validated();

        $data['photo'] = $request->hasFile('photo') 
            ? MediaHandler::upload($request->file('photo'), 'supervisor/photos')
            : null;

        $data['type'] = UserType::SUPERVISOR;
        $supervisor  = User::create($data);
        return $this->apiResponseStored(new UserResource($supervisor));
    }

    public function show(User $supervisor): JsonResponse
    {
        if (!$supervisor || $supervisor->type !== UserType::SUPERVISOR->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }
        return $this->apiResponseShow(new UserResource($supervisor));
    }

    public function update(UpdateUserRequest $request, User $supervisor): JsonResponse
    {
        if (!$supervisor || $supervisor->type !== UserType::SUPERVISOR->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }
        $data = $request->validated();
        $data['photo'] = $request->hasFile('photo')
            ? MediaHandler::updateMedia($request->file('photo'), 'supervisor/photos', $supervisor->photo_path)
            : $supervisor->photo;
        $supervisor->update($data);
        return $this->apiResponseUpdated(new UserResource($supervisor));
    }

    public function destroy(User $supervisor): JsonResponse
    {
        if (!$supervisor || $supervisor->type !== UserType::SUPERVISOR->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }
        $supervisor->photo ? MediaHandler::deleteMedia($supervisor->photo_path) : null;
        $supervisor->delete();
        return $this->apiResponseDeleted();
    }
}
