<?php

namespace App\Http\Controllers\Api\V1\Student;

use App\Http\Controllers\Controller;
use App\Http\Requests\Student\StudentProfile\UpdateStudentProfileRequest;
use App\Http\Resources\User\UserResource;
use App\Http\Traits\MediaHandler;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;

class StudentProfileController extends Controller
{
    use MediaHandler;

    /**
     * Get authenticated student's enrolled rounds.
     *
     * Retrieves enrolled rounds with related courses and performance data.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function showProfile(): JsonResponse
    {
        $student = auth()->user();
        return $this->apiResponseShow(new UserResource($student));
    }

    /**
     * Update authenticated student's profile.
     *
     * Validates and updates profile data, including password and photo.
     *
     * @param  UpdateStudentProfileRequest  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateProfile(UpdateStudentProfileRequest $request): JsonResponse
    {
        $student = auth()->user();
        $data = $request->validated();
        // Handle password update
        if (!empty($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        }
        $data['photo'] = $request->hasFile('photo')
            ? MediaHandler::updateMedia($request->file('photo'), 'student/photos', $student->photo_path)
            : $student->photo;
        $student->update($data);
        return $this->apiResponseUpdated(new UserResource($student));
    }
}
