<?php

namespace App\Http\Controllers\Api\V1\Student;

use App\Enums\DefineStatus;
use App\Http\Controllers\Controller;
use App\Http\Requests\Student\StudentTaskSubmission\TaskSubmissionRequest;
use App\Http\Resources\Lecture\LectureResource;
use App\Http\Resources\Round\RoundResource;
use App\Http\Resources\TaskFeedback\TaskFeedbackResource;
use App\Http\Resources\TaskSubmission\TaskSubmissionResource;
use App\Models\Lecture;
use App\Models\Round;
use App\Models\TaskFeedback;
use App\Models\TaskSubmission;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class StudentEnrollmentController extends Controller
{
    /**
     * Get authenticated student's enrolled rounds.
     *
     * Retrieves enrolled rounds with related courses and performance data.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function showStudentEnrollments(): JsonResponse
    {
        $user = auth()->user();

        $enrolledRounds = $user->enrolledRounds()
            ->wherePivot('status', DefineStatus::ACTIVE)
            ->with([
                'course',
                'myCourseEnrollment:id,round_id,user_id,coupon_amount'
            ])
            ->select(['rounds.id', 'rounds.name', 'rounds.course_id', 'rounds.start_date', 'rounds.round_content','rounds.type','rounds.link_url'])
            ->get();


        if ($enrolledRounds->isEmpty()) {
            return $this->apiResponseShow([]);
        }
        $enrolledRounds->each(function ($round) use ($user) {
            if ($round) {
                $round->total_task_degrees = $user->getTotalTaskDegreesForRound($round);
                // $round->total_user_round_evaluations = $round->getUserTotalTaskDegree($user);
                $round->total_user_points            = $round->getUserPoints($user);
            }
        });

        return $this->apiResponseShow(RoundResource::collection($enrolledRounds));
    }

    /**
     * Show details of a specific enrolled round.
     *
     * Ensures the student is enrolled before retrieving round details with lectures and evaluations.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  Round  $round
     * @return \Illuminate\Http\JsonResponse
     */
    public function showStudentEnrollmentDetails(Round $round): JsonResponse
    {
        $user = auth()->user();
        // Check if user is enrolled in this round and their status is active
        abort_if(!$user->studentRounds()->wherePivot('status', DefineStatus::ACTIVE)->where('round_id', $round->id)->exists(), 404);
        $lectures = Lecture::where('round_id', $round->id)
        ->with([
            'studentTaskSubmission.feedbacks.createdBy',
            'studentEvaluation',
            'taskSubmissions',
            'lectureTasks',
            'lectureVideos'
            // 'studentTaskSubmission',
            ])
            ->whereDate('start_time', '<=', now()->toDateString())
            ->orderByDesc('start_time')
        ->get();
        return $this->apiResponseShow(LectureResource::collection($lectures));
    }

    public function studentTaskSubmission(TaskSubmissionRequest $request, Round $round, Lecture $lecture): JsonResponse
    {
        $user = auth()->user();
        $data = $request->validated();

        // Check if user is enrolled in this round and their status is active
        abort_if(!$user->studentRounds()->wherePivot('status', DefineStatus::ACTIVE)->where('round_id', $round->id)->exists(), 404);

        // Ensure the lecture belongs to the round
        abort_if(!$round->lectures()->where('id', $lecture->id)->exists(), 404);

        TaskSubmission::where('user_id', $user->id)
            ->select('id', 'lecture_id', 'user_id', 'task_url', 'updated_at')
            ->updateOrCreate(
            [
                'user_id' => $user->id,
                'lecture_id' => $lecture->id
            ],
            [
                'task_url' => $data['task_url'],
            ]
        );

        $lecture->load(['taskSubmissions' => function($query) use ($user) {
            $query->where('user_id', $user->id)
                ->select('id', 'task_url', 'lecture_id', 'user_id', 'updated_at');
        },'lectureVideos']);

        return $this->apiResponseUpdated(new LectureResource($lecture));
    }

    public function getLectureDetails(Lecture $lecture){
        return $this->apiResponseShow(new LectureResource($lecture->load('lectureVideos')));
    }
    public function getCountUnReadFeedback(){
        $user = auth()->user();
        $unReadFeedback=TaskFeedback::whereHas('taskSubmission',function($q)use($user){
            $q->where('user_id',$user->id);
        })->where('read',false)->count();
        return $this->apiResponseShow([
            'unReadFeedbacks' => $unReadFeedback
        ]);
    }
    public function makeAsRead(TaskFeedback $taskFeedback){
        $taskFeedback->read = true;
        $taskFeedback->save();
        return $this->apiResponseUpdated(new TaskFeedbackResource($taskFeedback));
    }

    public function makeAllRead(TaskSubmission $taskSubmission){
        $taskSubmission->feedbacks()->where('read', false)->update(['read'=>true]);
        return $this->apiResponseUpdated(new TaskSubmissionResource($taskSubmission->load('feedbacks')));
    }
}
