<?php

namespace App\Http\Controllers\Api\V1\Student;

use App\Enums\UserType;
use App\Http\Controllers\Controller;
use App\Http\Requests\Student\StudentAuth\StudentLoginRequest as StudentLoginRequest;
use App\Http\Resources\User\UserResource;
use App\Http\Traits\ApiResponse;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;
use App\Http\Traits\ApiException;
use Illuminate\Http\Request;

class StudentAuthController extends Controller
{
    use ApiResponse;

    /**
     * Logs an existing admin in.
     * - Validates the incoming login credentials.
     * - Attempts to authenticate the user based on the provided email and password.
     * - Ensure that the authenticated user is admin.
     * - Generates an API token for the user.
     * - If authentication fails, returns an error response indicating invalid credentials.
     */
    public function login(StudentLoginRequest $request): JsonResponse
    {
        $request->validated();

        $user = User::students()
            ->where('email', $request->email)
            ->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return ApiException::apiResponse([], __('main.incorrect_credentials'), 401);
        }

        $token = $user->createToken('UserToken', ['type:student'])->plainTextToken;

        $user->update([
            'is_open'=>true,
            'last_activity'=>now(),
            'last_logout_at'=>null
        ]);

        return ApiResponse::apiResponse([
            'token' => $token,
            'user' => new UserResource($user->load('installments')),
        ], __('main.logged_in'), 200);
    }

    /**
     * Logs out the authenticated user.
     * - Deletes the current access token of the authenticated user.
     */
    public function logout(Request $request): JsonResponse
    {
        $request->user()->currentAccessToken()->delete();
        $request->user()->update([
            'is_open'=>false,
            'last_logout_at'=>now()
        ]);
        return ApiResponse::apiResponse([],__('main.logged_out'), 200);
    }
}
