<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\UserType;
use App\Http\Controllers\Controller;
use App\Http\Requests\RoundSupervisor\AddSupervisorRequest;
use App\Http\Requests\RoundSupervisor\ChangeSupervisorStatusRequest;
use App\Http\Resources\RoundSupervisor\RoundSupervisorResource;
use App\Http\Resources\User\UserResource;
use App\Models\Round;
use App\Models\RoundSupervisor;
use App\Models\User;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class RoundSupervisorController extends Controller
{
    /**
     * Add supervisors to a specific round.
     * Synchronizes the provided supervisor IDs with the round.
     *
     * @param  AddSupervisorRequest  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addSupervisor(AddSupervisorRequest $request)
    {
        $data = $request->validated();

        $round = Round::select('id', 'name')->findOrFail($data['round_id']);

        $round->supervisors()->syncWithoutDetaching($data['user_ids']);

        return $this->apiResponseStored(new RoundSupervisorResource($round));
    }

    /**
     * Assign a round to multiple supervisors.
     * Ensures each supervisor is valid and attaches the round without detaching existing ones.
     *
     * @param  AddSupervisorRequest  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addRoundToSupervisors(AddSupervisorRequest $request)
    {
        $data = $request->validated();

        $round = Round::findOrFail($data['round_id']);

        foreach ($data['user_ids'] as $userId) {
            $supervisor = User::where('id', $userId)
                        ->where('type', UserType::SUPERVISOR->value)
                        ->firstOrFail();

            $supervisor->supervisorRounds()->syncWithoutDetaching([$round->id]);
        }

        $supervisors = User::with('supervisorRounds')
            ->whereIn('id', $data['user_ids'])
            ->get();

        return $this->apiResponseStored(UserResource::collection($supervisors));
    }

    /**
     * Change the status of a specific supervisor in a round.
     *
     * Validates the status update and ensures the supervisor belongs to the specified round.
     *
     * @param  ChangeSupervisorStatusRequest  $request
     * @param  Round  $round
     * @param  User  $supervisor
     * @return \Illuminate\Http\JsonResponse
     */
    public function changeSupervisorStatus(ChangeSupervisorStatusRequest $request, Round $round, User $supervisor)
    {
        $validated = $request->validated();
        $round->load('supervisors');
        abort_if(!$round->supervisors->contains($supervisor), 404);

        $round->supervisors()->updateExistingPivot($supervisor->id, [
            'status' => $validated['status'],
        ]);

        $supervisor->load(['supervisorRounds' => function($query) use ($round) {
            $query->where('round_id', $round->id);
        }]);

        return $this->apiResponseStored(new UserResource($supervisor));
    }

    /**
     * Retrieve a paginated list of supervisors in a specific round.
     *
     * Applies dynamic filters and pagination.
     *
     * @param  Round  $round
     * @return AnonymousResourceCollection
     */
    public function showSupervisorsInRound(Round $round): AnonymousResourceCollection
    {
        $supervisors = RoundSupervisor::where('round_id', $round->id)->pluck('user_id');
        $supervisors = User::whereIn('id', $supervisors)
            ->with([
                'supervisorRounds' => fn($query) => $query->where('round_id', $round->id),
            ])
            ->useFilters()
            ->dynamicPaginate();
        return UserResource::collection($supervisors);
    }
}
