<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\InstallmentStatusEnum;
use App\Enums\PaymentMethodEnum;
use App\Enums\PaymentTypeEnum;
use App\Exports\UsersExport;
use App\Helpers\ValidationMessageHelper;
use App\Imports\ImportLecture;
use App\Imports\UsersImport;
use App\Models\Round;
use App\Http\Traits\ApiResponse;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use App\Http\Resources\Round\RoundResource;
use App\Http\Requests\Round\CreateRoundRequest;
use App\Http\Requests\Round\UpdateRoundRequest;
use App\Http\Resources\Lecture\LectureResource;
use App\Models\Evaluation;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Maatwebsite\Excel\Facades\Excel;

class RoundController extends Controller
{
    use ApiResponse;

    public function index(): AnonymousResourceCollection
    {
        $rounds = Round::with([
            'course:id,title',
            'instructor',
            'lectures',
            'students:id',
            'branch',
            // 'transactions',
            // 'installments',
            'notices',
        ])
        ->useFilters()
        ->latest()
        ->dynamicPaginate(50);
        return RoundResource::collection($rounds);
    }

    public function store(CreateRoundRequest $request): JsonResponse
    {
        $data  = $request->validated();
        $round = Round::create($data);
        // isset($data['student_id']) && $round->students()->attach($data['student_id']);
        $this->loadRelations($round);
        // $round->students_count = $round->students()->count();
        // $round->save();
        return $this->apiResponseStored(new RoundResource($round->load('branch')));
    }

    public function show(Round $round): JsonResponse
    {
        $round->load('course:id,title', 'instructor:id,name,email','students:id');
        return $this->apiResponseShow(new RoundResource($round));
    }

    public function update(UpdateRoundRequest $request, Round $round): JsonResponse
    {
        $data = $request->validated();
        $round->update($data);

        $round->additional_instructors = $data['additional_instructors'] ?? null;
        $round->save();

        // isset($data['student_id']) && $round->students()->sync($data['student_id']);
        $this->loadRelations($round);
        // $round->students_count = $round->students()->count();
        // $round->save();

        return $this->apiResponseUpdated(new RoundResource($round->load('branch')));
    }

    public function destroy(Round $round): JsonResponse
    {
        if (($round->students_count > 0)) {
            return $this->apiResponse([], ValidationMessageHelper::cannotDelete('round', 'students'), 422);
        }

        $round->delete();
        return $this->apiResponseDeleted();
    }


    public function lectures(Round $round): AnonymousResourceCollection
    {
        $lectures = $round->lectures()->orderByDesc('id')->useFilters()->with('zoomMeetings:id,lecture_id,start_time,duration','attachments','taskSubmissions','studentEvaluation','studentTaskSubmission','lectureTasks','lectureVideos','round.students')->paginate(50);
        return LectureResource::collection($lectures);
    }

    protected function loadRelations(Round $round): Round
    {
        return $round->load([
            'course:id,title',
            'instructor:id,name,email',
            'students:id,name,email',
        ]);
    }


    public function importStudents(Request $request)
    {
        $request->validate([
            'round_id' => 'required|exists:rounds,id',
            'file'     => 'required|file|mimes:xlsx,csv,xls'
        ]);

        $usersImport = new UsersImport();
        $usersImport->setRoundId($request->round_id);

        Excel::import($usersImport, $request->file('file'));

        $invalidRows=$usersImport->getInvalidRows();

        if(!empty($invalidRows)){
            return $this->apiResponseFailed($invalidRows);
        }

        return $this->apiResponseShow([]);
    }
    public function exportStudents(Request $request)
    {
        $request->validate([
            'student_ids' => 'array|nullable',
            'round_id' => 'nullable|exists:rounds,id'
        ]);
        return Excel::download(new UsersExport($request->student_ids,$request->round_id), 'students.xlsx');
    }

    public function addLinkToRound(Request $request, Round $round){
        $request->validate([
            'link_url' => 'required|url'
        ]);
        $round->update([
            'link_url' => $request->link_url
        ]);
        return $this->apiResponseUpdated([]);
    }

    public function mergeContent(Round $round){
        $round->update([
            'round_content'=>$round->course?->course_content
        ]);
        return $this->apiResponseUpdated($round->round_content);
    }

    public function editRoundContent(Request $request, Round $round){
        $request->validate([
            'round_content' => 'required'
        ]);
        $round->update([
            'round_content' => $request->round_content
        ]);

        $round->load([
            'course:id,title',
            'instructor',
            'lectures',
            'students:id',
            'branch',
            'notices',
        ]);

        return $this->apiResponseUpdated(new RoundResource($round));
    }
    public function getStats(Round $round)
    {
        $round->load(['students', 'lectures', 'installments', 'transactions', 'courseEnrollments']);

        $studentsCount = $round->students()->count();
        $lecturesCount = $round->lectures()->count();
        $installmentsCount = $round->installments()->count();
        $transactionsCount = $round->transactions()->count();
        $transactionsTotal = $round->transactions()->sum('amount');

        $totalInstallmentAmount = $round->installments->sum('amount');
        $totalPaidInstallmentsAmount = $round->installments->where('status', InstallmentStatusEnum::PAID->value)->sum('amount');
        $totalUnpaidInstallmentsAmount = $totalInstallmentAmount - $totalPaidInstallmentsAmount;

        $totalEnrollmentDiscount = $round->courseEnrollments->sum('coupon_amount') ?? 0;
        $totalEnrollmentAmount = $round->courseEnrollments->sum('total_amount') ?? 0;

        $expectedAmount = $totalEnrollmentAmount-$totalEnrollmentDiscount;

        $lateInstallments = $round->installments
            ->where('status', '!=', InstallmentStatusEnum::PAID->value)
            ->where('due_date', '<', now());
        $lateInstallmentsCount = $lateInstallments->count();
        $lateInstallmentsAmount = $lateInstallments->sum('amount');

        $totalTransactionAmountCash = $round->transactions()
            ->where('transactions.payment_method', PaymentTypeEnum::CASH->value)
            ->sum('transactions.amount');

        $totalAllAmountAllPaidOrNot = $totalInstallmentAmount + $totalTransactionAmountCash;
        $totalAllAmount = $totalPaidInstallmentsAmount + $totalTransactionAmountCash;

        $totalAllAmountRate = $totalAllAmountAllPaidOrNot > 0
            ? round(($totalAllAmount / $totalAllAmountAllPaidOrNot) * 100, 2)
            : 0;

        $paymentRate = $totalInstallmentAmount > 0
            ? round(($totalPaidInstallmentsAmount / $totalInstallmentAmount) * 100, 2)
            : 0;

        $lateInstallmentsRate = $totalInstallmentAmount > 0
            ? round(($lateInstallmentsAmount / $totalInstallmentAmount) * 100, 2)
            : 0;

        $paymentMethods = PaymentTypeEnum::cases();
        $transactionsCountByMethod = [];
        $transactionsSumByMethod = [];
        foreach ($paymentMethods as $method) {
            $transactionsCountByMethod[$method->name] = $round->transactions()
                ->where('transactions.payment_method', $method->value)
                ->count();
            $transactionsSumByMethod[$method->name] = $round->transactions()
                ->where('transactions.payment_method', $method->value)
                ->sum('transactions.amount');
        }

        $attendanceRate = $round->getAttendanceRate();
        $absenceCount = $round->getAbsence($round->id);

        $taskSubmissionRate = $round->getTaskSubmissionRate();
        $taskDegreeSummary = $round->taskSubmissionsDegreeSummary();

        $stats = [
            'general' => [
                'students_count' => $studentsCount,
                'lectures_count' => $lecturesCount,
                'installments_count' => $installmentsCount,
                'transactions_count' => $transactionsCount,
                'course_enrollments_count' => $round->courseEnrollments()->count(),
            ],
            'financial' => [
                'group_amount' => $totalEnrollmentAmount,
                'total_discount' => $totalEnrollmentDiscount,
                'expected_amount' => $expectedAmount,
                'total_transactions' => $transactionsTotal,
                'total_paid_installments_amount' => $totalPaidInstallmentsAmount,
                'total_unpaid_installments_amount' => $totalUnpaidInstallmentsAmount,
                'total_installments_amount' => $totalInstallmentAmount,
                'late_installments_count' => $lateInstallmentsCount,
                'late_installments_amount' => $lateInstallmentsAmount,
                'late_installments_rate_percentage' => $lateInstallmentsRate,
                'payment_rate_percentage' => $paymentRate,
                'totalAllAmountRate' => $totalAllAmountRate,
                'totalAllAmount' => $totalAllAmount,
                'totalAllAmountAllPaidOrNot' => $totalAllAmountAllPaidOrNot,
                'transactions_count_by_method' => $transactionsCountByMethod,
                'transactions_sum_by_method' => $transactionsSumByMethod,
            ],
            'attendance' => [
                'attendance_rate_percentage' => $attendanceRate,
                'absence_count' => $absenceCount,
            ],
            'tasks' => [
                'task_submission_rate_percentage' => $taskSubmissionRate,
                'task_submissions_degree_summary' => $taskDegreeSummary,
            ],
        ];

        return $this->apiResponseShow($stats);
    }

    public function getGroupReport(Round $round)
    {
        $round->load([
            'lectures' => function($query) {
                $query->with([
                    'evaluations',
                    'lectureTasks',
                    'taskSubmissions'
                ]);
            },
            'students',
            'transactions:amount',
            'installments:amount,course_enrollment_id',
            'courseEnrollments:total_amount,round_id,id'
        ]);

        $totalTransactions = $round->transactions->sum('amount');
        $totalInstallments = $round->installments->sum('amount');
        $totalExpectedAmount = $round->courseEnrollments->sum('total_amount');

        $installmentCounts = $round->installments->groupBy('course_enrollment_id')
            ->map->count();

        $numberOfInstallments = $installmentCounts->isNotEmpty()
            ? round($installmentCounts->avg(), 1)
            : 0;

        $studentIds = $round->students ? $round->students->pluck('id')->toArray() : [];
        $studentCount = count($studentIds);
        $lectureCount = $round->lectures ? $round->lectures->count() : 0;

        $lectureCharts = [];
        $totalPossibleAttendances = $lectureCount * $studentCount;
        $attendanceCount = 0;
        $taskSubmissionCount = 0;
        $totalPossibleTasks = 0;
        $totalTaskDegrees = 0;
        $maxPossibleDegrees = 0;

        if ($round->lectures) {
            foreach ($round->lectures as $lecture) {
                $evaluations = $lecture->evaluations ? $lecture->evaluations : collect([]);
                $taskSubmissions = $lecture->taskSubmissions ? $lecture->taskSubmissions : collect([]);
                $lectureTasks = $lecture->lectureTasks ? $lecture->lectureTasks : collect([]);

                $lectureAttendance = $evaluations
                    ->whereIn('user_id', $studentIds)
                    ->where('is_attend', true)
                    ->count();

                $attendanceCount += $lectureAttendance;
                $attendancePercentage = $studentCount > 0
                    ? round(($lectureAttendance / $studentCount) * 100, 2)
                    : 0;

                $lectureTaskSubmissions = $taskSubmissions
                    ->whereIn('user_id', $studentIds)
                    ->count();

                $lectureTasksCount = $lectureTasks->count() * $studentCount;
                $totalPossibleTasks += $lectureTasksCount;
                $taskSubmissionCount += $lectureTaskSubmissions;

                $submissionPercentage = $lectureTasksCount > 0
                    ? round(($lectureTaskSubmissions / $lectureTasksCount) * 100, 2)
                    : 0;

                $lectureDegreeSum = $taskSubmissions
                    ->whereIn('user_id', $studentIds)
                    ->sum('task_degree');

                $totalTaskDegrees += $lectureDegreeSum;
                $averageDegree = $lectureTaskSubmissions > 0
                    ? round($lectureDegreeSum / $lectureTaskSubmissions, 2)
                    : 0;

                $lectureCharts[] = [
                    'lecture_name' => $lecture->name ?? 'N/A',
                    'attendance_percentage' => $attendancePercentage,
                    'submission_percentage' => $submissionPercentage,
                    'average_degree' => $averageDegree,
                    'attendance_count' => $lectureAttendance,
                    'submission_count' => $lectureTaskSubmissions,
                    'total_possible_submissions' => $lectureTasksCount
                ];
            }
        }

        $overallAttendance = $totalPossibleAttendances > 0
            ? round(($attendanceCount / $totalPossibleAttendances) * 100, 2)
            : 0;

        $overallSubmission = $totalPossibleTasks > 0
            ? round(($taskSubmissionCount / $totalPossibleTasks) * 100, 2)
            : 0;

        $overallAverageDegree = $taskSubmissionCount > 0
            ? round($totalTaskDegrees / $taskSubmissionCount, 2)
            : 0;

        $lectureCollection = collect($lectureCharts);

        $roundInfo = [
            'id' => $round->id,
            'name' => $round->name,
            'status' => $round->status->value,
            'start_date' => $round->start_date,
            'end_date' => $round->end_date,
            'course' => [
                'id' => $round->course_id,
                'name' => $round->course->name ?? 'N/A',
            ],
            'instructor' => [
                'id' => $round->instructor_id,
                'name' => $round->instructor->name ?? 'N/A',
                'email' => $round->instructor->email ?? 'N/A',
            ],
            'branch' => [
                'id' => $round->branch_id,
                'name' => $round->branch->name ?? 'N/A',
            ],
            'schedule' => [
                'total_lectures' => $lectureCount,
                'completed_lectures' => $round->lectures->where('date', '<=', now())->count(),
                'upcoming_lectures' => $round->lectures->where('date', '>', now())->count(),
            ],
            'enrollment' => [
                'max_students' => $round->max_student,
                'current_students' => $studentCount,
                'enrollment_percentage' => $round->max_student > 0
                    ? round(($studentCount / $round->max_student) * 100, 2)
                    : 0,
            ]
        ];

        return $this->apiResponse([
            'round_info' => $roundInfo,
            'summary' => [
                'total_students' => $studentCount,
                'total_lectures' => $lectureCount,
                'overall_attendance_percentage' => $overallAttendance,
                'overall_submission_percentage' => $overallSubmission,
                'overall_average_degree_percentage' => $overallAverageDegree,
                'total_attendance' => $attendanceCount,
                'total_submissions' => $taskSubmissionCount,
                'total_possible_attendances' => $totalPossibleAttendances,
                'total_possible_submissions' => $totalPossibleTasks,
                'financial' => [
                    'total_transactions' => (float) $totalTransactions,
                    'total_installments' => (float) $totalInstallments,
                    'total_expected_amount' => (float) $totalExpectedAmount,
                    'number_of_installments' => (float) $numberOfInstallments,
                    'payment_completion_rate' => $totalExpectedAmount > 0
                        ? (float) number_format(($totalTransactions / $totalExpectedAmount) * 100, 2)
                        : 0.0,
                    'remaining_balance' => (float) max(0, $totalExpectedAmount - $totalTransactions),
                ],
            ],
            // 'lecture_metrics' => $lectureCharts,
            'charts' => [
                'attendance' => [
                    'labels' => $lectureCollection->pluck('lecture_name'),
                    'data' => $lectureCollection->pluck('attendance_percentage'),
                    'counts' => $lectureCollection->pluck('attendance_count'),
                    'total_students' => $studentCount,
                ],
                'submissions' => [
                    'labels' => $lectureCollection->pluck('lecture_name'),
                    'data' => $lectureCollection->pluck('submission_percentage'),
                    'counts' => $lectureCollection->pluck('submission_count'),
                    'total_possible' => $lectureCollection->pluck('total_possible_submissions'),
                ],
                'degrees' => [
                    'labels' => $lectureCollection->pluck('lecture_name'),
                    'average_degrees' => $lectureCollection->pluck('average_degree'),
                ]
            ]
        ]);
    }

    public function addSkillsStudents(Round $round){
        $course = $round->course;

        foreach($round->students as $student){
            if($this->checkScore($student, $round)){
                $student->skills()->create([
                    'title' => $course->title,
                    'course_id' => $course->id,
                ]);
            }
        }

        return $this->apiResponseUpdated([]);
    }


    public function checkScore(User $user, Round $round){
        $score = min(100, round((($this->getAttendanceRate($round, $user) + $round->getTaskSubmissionRateByUser()) / 2) + $user->bonus, 2));

        return $score >= 80;
    }

    public function getAttendanceRate(Round $round, User $user)
    {
        $lectureIds = $round->lectures()->pluck('lectures.id');

        $totalLectures = $lectureIds->count();

        $actualAttendances = Evaluation::whereIn('lecture_id', $lectureIds)
            ->where('user_id', $user->id)
            ->where('is_attend', true)
            ->count();

        return round(($actualAttendances / $totalLectures) * 100, 2);
    }

}
