<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\Role\RoleResource;
use App\Http\Traits\ApiResponse;
use App\Models\Permission;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use App\Http\Resources\Role\PermissionResource;

class RoleController extends Controller
{
    use ApiResponse;

    // public function __construct()
    // {
    //     $this->middleware('permission:role-list', ['only' => ['index', 'show']]);
    //     $this->middleware('permission:role-create', ['only' => ['store']]);
    //     $this->middleware('permission:role-edit', ['only' => ['update']]);
    //     $this->middleware('permission:role-delete', ['only' => ['destroy']]);
    // }

    /**
     * Display a paginated listing of roles.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        $roles = Role::withCount('permissions')
        ->with('permissions')
            ->orderBy('id', 'DESC')
            ->dynamicPaginate();

        return RoleResource::collection($roles);
    }

    public function getPermission(){
        $permissions = Permission::all();
        return $this->apiResponse(PermissionResource::collection($permissions));
    }

    /**
     * Get all permissions for role creation/editing
     *
     * @return \Illuminate\Http\JsonResponse
     */
    // public function getPermission()
    // {
    //     $permissions = Permission::orderBy('name')->get()->groupBy(function ($item) {
    //         return explode('-', $item->name)[0];
    //     });

    //     return $this->apiResponse([
    //         'permissions' => $permissions,
    //         'grouped' => true
    //     ]);
    // }

    /**
     * Store a newly created role in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:roles,name',
            'permission' => 'required|array',
            'permission.*' => 'exists:permissions,id'
        ]);

        $role = null;

        DB::transaction(function () use (&$role, $validated) {
            $role = Role::create([
                'name' => $validated['name'],
            ]);

            $role->permissions()->sync($validated['permission']);
        });

        return $this->apiResponseStored(
            new RoleResource($role->load('permissions'))
        );
    }


    /**
     * Display the specified role with its permissions.
     *
     * @param  Role  $role
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Role $role)
    {
        $role->load('permissions');

        return $this->apiResponseShow(new RoleResource($role));
    }

    /**
     * Update the specified role in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  Role  $role
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, Role $role)
    {
        $validated = $request->validate([
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('roles')->ignore($role->id),
            ],
            'permission' => 'required|array',
            'permission.*' => 'exists:permissions,id',
        ]);

        DB::transaction(function () use ($validated, $role) {
            $role->update([
                'name' => $validated['name'],
            ]);

            $role->permissions()->sync($validated['permission']);
        });

        return $this->apiResponseUpdated(
            new RoleResource($role->fresh()->load('permissions'))
        );
    }
    /**
     * Remove the specified role from storage.
     *
     * @param  Role  $role
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Role $role)
    {
        if ($role->name === 'admin') {
            return $this->apiResponseFailed('Cannot delete admin role');
        }

        if ($role->users()->exists()) {
            return $this->apiResponseFailed('Cannot delete role assigned to users');
        }

        $role->permissions()->detach();
        $role->delete();

        return $this->apiResponseDeleted();
    }
}
