<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\Evaluation;
use App\Models\Round;
use App\Models\TaskFeedback;
use App\Models\TaskSubmission;
use App\Models\User;
use App\Services\ReportService;
use Illuminate\Http\Request;


class ReportController extends Controller
{
    private ReportService $reportService;

    public function __construct(ReportService $reportService)
    {
        $this->reportService = $reportService;
    }


    public function counts()
    {
    $data = $this->reportService->getCounts();
    return $this->apiResponse($data, __('main.success_query'), 200);
    }

    public function roundsReport(Request $request)
    {
        $filters = $request->only(['year', 'course_id', 'status', 'from_date', 'to_date']);
        $report = $this->reportService->generateReport(Round::query(), $filters);
        return $this->apiResponse($report, __('main.success_query'), 200);
    }

    public function coursesReport(Request $request)
    {
        $filters = $request->only(['year', 'status', 'from_date', 'to_date']);
        $report = $this->reportService->generateReport(Course::query(), $filters);
        return $this->apiResponse($report, __('main.success_query'), 200);
    }

    public function studentsReport(Request $request)
    {
        $filters = $request->only(['year', 'status', 'from_date', 'to_date']);
        $query   = User::students();
        if ($request->has('course_id')) {
            $courseId = $request->course_id;
            $query->whereHas('enrolledCourses', function ($courseQuery) use ($courseId) {
                $courseQuery->where('course_id', $courseId);
            });
        }

        $report = $this->reportService->generateReport($query, $filters);

        return $this->apiResponse($report, __('main.success_query'), 200);
    }

    public function getGroupReport(Round $round): array
    {
        $round = Round::with(['lectures', 'students'])->findOrFail($round->id);

        $lectureCount = $round->lectures->count();
        $studentCount = $round->students->count();

        $lectureIds = $round->lectures->pluck('id');
        $studentIds = $round->students->pluck('id');

        $totalPossibleAttendances = $lectureCount * $studentCount;
        $attendanceCount = 0;
        $absenceCount = 0;

        if ($totalPossibleAttendances > 0) {
            $evaluations = Evaluation::whereIn('lecture_id', $lectureIds)
                ->whereIn('user_id', $studentIds)
                ->get();

            $attendanceCount = $evaluations->where('is_attend', true)->count();
            $absenceCount = $evaluations->where('is_attend', false)->count();
        }

        $taskSubmissionCount = 0;
        $totalPossibleTasks = 0;

        $lectures = $round->lectures()->with('lectureTasks')->get();

        foreach ($lectures as $lecture) {
            if ($lecture->relationLoaded('lectureTasks') && $lecture->lectureTasks) {
                $totalPossibleTasks += $studentCount;
                $taskSubmissionCount += TaskSubmission::where('lecture_id', $lecture->id)
                    ->whereIn('user_id', $studentIds)
                    ->count();
            }
        }

        $feedbackCount = TaskFeedback::whereHas('taskSubmission', function($query) use ($lectureIds) {
                $query->whereIn('lecture_id', $lectureIds);
            })
            ->count();

        $attendancePercentage = $totalPossibleAttendances > 0
            ? round(($attendanceCount / $totalPossibleAttendances) * 100, 2)
            : 0;

        $absencePercentage = $totalPossibleAttendances > 0
            ? round(($absenceCount / $totalPossibleAttendances) * 100, 2)
            : 0;

        $taskSubmissionPercentage = $totalPossibleTasks > 0
            ? round(($taskSubmissionCount / $totalPossibleTasks) * 100, 2)
            : 0;

        return [
            'lectures_count' => $lectureCount,
            'students_count' => $studentCount,
            'attendance_count' => $attendanceCount,
            'absence_count' => $absenceCount,
            'attendance_percentage' => $attendancePercentage,
            'absence_percentage' => $absencePercentage,
            'task_submission_percentage' => $taskSubmissionPercentage,
            'feedbacks_count' => $feedbackCount,
        ];
    }

    public function getStudentReport($studentId)
    {
        $student = User::with(['enrolledRounds.lectures', 'evaluations', 'taskSubmissions.taskFeedbacks'])
            ->findOrFail($studentId);

        $rounds = $student->enrolledRounds;
        $allRoundsData = [];

        foreach ($rounds as $round) {
            $lectureIds = $round->lectures->pluck('id');
            $lectureCount = $round->lectures->count();

            $evaluations = $student->evaluations()
                ->whereIn('lecture_id', $lectureIds)
                ->get();

            $attendanceCount = $evaluations->where('is_attend', true)->count();
            $absenceCount = $evaluations->where('is_attend', false)->count();

            $taskSubmissionCount = $student->taskSubmissions()
                ->whereIn('lecture_id', $lectureIds)
                ->count();

            $totalPossibleTasks = $round->lectures()->has('lectureTasks')->count();

            $feedbackCount = $student->taskSubmissions()
                ->whereIn('lecture_id', $lectureIds)
                ->has('taskFeedbacks')
                ->count();

            $allRoundsData[] = [
                'round_id' => $round->id,
                'round_name' => $round->name,
                'lectures_count' => $lectureCount,
                'attendance_count' => $attendanceCount,
                'absence_count' => $absenceCount,
                'attendance_percentage' => $lectureCount > 0 ? round(($attendanceCount / $lectureCount) * 100, 2) : 0,
                'task_submission_count' => $taskSubmissionCount,
                'total_possible_tasks' => $totalPossibleTasks,
                'task_submission_percentage' => $totalPossibleTasks > 0 ? round(($taskSubmissionCount / $totalPossibleTasks) * 100, 2) : 0,
                'feedbacks_count' => $feedbackCount,
            ];
        }

        $totals = [
            'total_lectures' => collect($allRoundsData)->sum('lectures_count'),
            'total_attendance' => collect($allRoundsData)->sum('attendance_count'),
            'total_absence' => collect($allRoundsData)->sum('absence_count'),
            'total_task_submissions' => collect($allRoundsData)->sum('task_submission_count'),
            'total_possible_tasks' => collect($allRoundsData)->sum('total_possible_tasks'),
            'total_feedbacks' => collect($allRoundsData)->sum('feedbacks_count'),
        ];

        $totals['attendance_percentage'] = $totals['total_lectures'] > 0
            ? round(($totals['total_attendance'] / $totals['total_lectures']) * 100, 2)
            : 0;

        $totals['task_submission_percentage'] = $totals['total_possible_tasks'] > 0
            ? round(($totals['total_task_submissions'] / $totals['total_possible_tasks']) * 100, 2)
            : 0;

        return [
            'student' => [
                'id' => $student->id,
                'name' => $student->name,
                'email' => $student->email,
            ],
            'rounds' => $allRoundsData,
            'totals' => $totals,
        ];
    }

}
