<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\QuestionType;
use App\Http\Resources\LectureQuestionCase\LectureQuestionCaseResource;
use App\Models\LectureQuestion;
use App\Http\Traits\MediaHandler;
use Illuminate\Http\JsonResponse;
use App\Models\LectureQuestionCase;
use App\Http\Controllers\Controller;
use App\Http\Requests\LectureQuestion\CreateLectureQuestionCaseRequest;
use App\Http\Resources\LectureQuestion\LectureQuestionResource;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use App\Http\Requests\LectureQuestion\CreateLectureQuestionRequest;
use App\Http\Requests\LectureQuestion\UpdateLectureQuestionRequest;
use App\Http\Traits\NotificationTrait;
class LectureQuestionController extends Controller
{
    use MediaHandler, NotificationTrait;
    public function __construct()
    {
        //
    }

    public function index(): AnonymousResourceCollection
    {
        $lectureQuestions = LectureQuestion::useFilters()
        ->with(['options','case','tag'])
        ->dynamicPaginate();
        return LectureQuestionResource::collection($lectureQuestions);
    }

    public function store(CreateLectureQuestionRequest $request): JsonResponse
    {
        $data = $request->validated();
        if($data['question_type'] == QuestionType::Listening->value){
            $data['listening_content'] = $this->upload($data['listening_content'],'lecture-questions');
        }
        $lectureQuestion = LectureQuestion::create($data);
            if(isset($data['options']) && !empty($data['options'])){
                $lectureQuestion->options()->createMany($data['options']);
            }

        $lectureQuestion->load(['options','case','lecture','tag']);
        $this->sendPushNotificationClientSide('admin','add_quiz',$lectureQuestion->lecture->round_id);
        return $this->apiResponseStored(new LectureQuestionResource($lectureQuestion));
    }

    public function show(LectureQuestion $lectureQuestion): JsonResponse
    {
        return $this->apiResponseShow( new LectureQuestionResource($lectureQuestion));
    }

    public function update(UpdateLectureQuestionRequest $request, LectureQuestion $lectureQuestion): JsonResponse
    {
        $data = $request->validated();
        if($data['question_type'] == QuestionType::Listening->value && isset($data['question_content'])){
            $data['listening_content'] = $this->updateMedia($data['listening_content'],'lecture-questions', $lectureQuestion->listening_content);
        }
        $lectureQuestion->update($data);
        if(isset($data['options'])){
            foreach ($data['options'] as $optionData) {
                if (isset($optionData['id'])) {
                    $lectureQuestion->options()->where('id', $optionData['id'])->update([
                        'option_content' => $optionData['option_content'],
                        'is_correct'     => $optionData['is_correct']
                    ]);
                } else {
                    $lectureQuestion->options()->create($optionData);
                }
            }
        }
        $lectureQuestion->load(['options','case','tag']);
        return $this->apiResponseUpdated(new LectureQuestionResource($lectureQuestion));
    }

    public function destroy(LectureQuestion $lectureQuestion): JsonResponse
    {
        $lectureQuestion->delete();
        return $this->apiResponseDeleted();
    }


    public function indexCases(): AnonymousResourceCollection
    {
        $cases = LectureQuestionCase::useFilters()->dynamicPaginate();
        return LectureQuestionCaseResource::collection($cases);
    }

    public function createCase(CreateLectureQuestionCaseRequest $request): JsonResponse
    {
        $data = $request->validated();
        $case = LectureQuestionCase::create($data);
        return $this->apiResponseStored(new LectureQuestionCaseResource($case));
    }

}
