<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\UserType;
use App\Http\Controllers\Controller;
use App\Http\Requests\User\CreateUserRequest;
use App\Http\Requests\User\UpdateUserRequest;
use App\Http\Resources\User\UserResource;
use App\Http\Traits\MediaHandler;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;


class InstructorController extends Controller
{
    use MediaHandler;

    public function index(): AnonymousResourceCollection
    {
        $instructors = User::Instructors()
            ->useFilters()
            ->latest()
            ->dynamicPaginate(50);
        return UserResource::collection($instructors);
    }

    public function store(CreateUserRequest $request): JsonResponse
    {
        $data = $request->validated();

        $data['photo'] = $request->hasFile('photo')
            ? MediaHandler::upload($request->file('photo'), 'instructor/photos')
            : null;

        $data['type'] = UserType::INSTRUCTOR;
        $instructor = User::create($data);
        return $this->apiResponseStored(new UserResource($instructor));
    }

    public function show(User $instructor): JsonResponse
    {
        if (!$instructor || $instructor->type !== UserType::INSTRUCTOR->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }

        return $this->apiResponseShow(new UserResource($instructor));
    }

    public function update(UpdateUserRequest $request, User $instructor): JsonResponse
    {
        if (!$instructor || $instructor->type !== UserType::INSTRUCTOR->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }
        $data = $request->validated();

        $data['photo'] = $request->hasFile('photo')
            ? MediaHandler::updateMedia($request->file('photo'), 'instructor/photos', $instructor->photo_path)
            : $instructor->photo;

        $instructor->update($data);
        return $this->apiResponseUpdated(new UserResource($instructor));
    }

    public function destroy(User $instructor): JsonResponse
    {
        if (!$instructor || $instructor->type !== UserType::INSTRUCTOR->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }

        $instructor->photo ? MediaHandler::deleteMedia($instructor->photo_path) : null;
        $instructor->delete();
        return $this->apiResponseDeleted();
    }
}
