<?php

namespace App\Http\Controllers\Api\V1\Instructor;

use App\Enums\FeedbackType;
use App\Http\Controllers\Controller;
use App\Http\Requests\Instructor\InstructorRound\UpdateLectureTaskRequest;
use App\Http\Requests\Lecture\GradeSubmissionRequest;
use App\Http\Requests\Lecture\StoreFeedbackRequest;
use App\Http\Resources\Lecture\LectureResource;
use App\Http\Resources\LectureTask\LectureTaskResource;
use App\Http\Resources\Project\ProjectResource;
use App\Http\Resources\TaskFeedback\TaskFeedbackResource;
use App\Http\Resources\TaskSubmission\TaskSubmissionResource;
use App\Http\Resources\User\UserResource;
use App\Http\Traits\EnsuresInstructorAccess;
use App\Http\Traits\MediaHandler;
use App\Models\Evaluation;
use App\Models\Lecture;
use App\Models\LectureTask;
use App\Models\Project;
use App\Models\Round;
use App\Models\TaskFeedback;
use App\Models\TaskSubmission;
use App\Models\User;
use Illuminate\Http\Request;

class InstructorRoundTaskController extends Controller
{
    use MediaHandler;
    use EnsuresInstructorAccess;

    public function addLectureTask(UpdateLectureTaskRequest $request, Round $round, Lecture $lecture)
    {
        $this->ensureLectureBelongsToRound($round, $lecture);

        $data                   = $request->validated();
        if(!empty($data['files_included'])) {
            $data['files_included'] = $this->handleTaskFiles($data['files_included']);
        }
        $lectureTask            = LectureTask::create($data);
        return $this->apiResponseStored(new LectureTaskResource($lectureTask));
    }

    public function updateLectureTask(UpdateLectureTaskRequest $request, Round $round, Lecture $lecture, LectureTask $lectureTask)
    {
        $this->ensureLectureBelongsToRound($round, $lecture);

        $data = $request->validated();
        if (isset($data['files_included'])) {
            // foreach ($lectureTask->files_included as $file) {
            //     $this->deleteMedia($file['path']);
            // }
            $this->deleteMedia($data['files_included']);
            $data['files_included'] = $this->handleTaskFiles($data['files_included']);
        }

        $lectureTask->task_link=null;
        $lectureTask->files_included=null;
        $lectureTask->save();

        $lectureTask->update($data);
        return $this->apiResponseUpdated(new LectureTaskResource($lectureTask));
    }

    public function handleTaskFiles($files)
    {
        // if (isset($files) && count($files) > 0) {
        //     $processedFiles = [];
        //     foreach ($files as $file) {
        //         $processedFiles[] = [
        //             'ext'  => $file->getClientOriginalExtension(),
        //             'path' => $this->upload($file, 'lecture-tasks')
        //         ];
        //     }
        // }
        return $this->upload($files, 'lecture-tasks');
    }

    public function taskSubmissions(Round $round, Lecture $lecture)
    {
        $this->ensureLectureBelongsToRound($round, $lecture);

        $students = $round->studentRounds()
            ->select('users.id', 'users.name')
            ->with([
                'taskSubmissions' => fn($query) => $query->where('lecture_id', $lecture->id),
                'evaluation' => function ($q) use ($lecture) {
                    $q->where('lecture_id', $lecture->id);
                },
            ])
            ->orderBy('users.created_at', 'desc')
            ->dynamicPaginate(50);

        return UserResource::collection($students);
    }

    public function gradeSubmission(GradeSubmissionRequest $request, Round $round, Lecture $lecture, TaskSubmission $task)
    {
        $instructor = auth()->user();
        $this->ensureTaskBelongsToLecture($round, $lecture, $task);

        $data = $request->validated();

        $task = TaskSubmission::where('lecture_id', $lecture->id)
            ->where('user_id', $data['user_id'])
            ->first();

        if ($task) {
            $task->update([
                'task_degree'  => $data['task_degree'],
                'is_corrected' => true,
                'graded_by'    => $instructor->id,
            ]);
        } else {
            $task = TaskSubmission::create([
                'lecture_id'   => $lecture->id,
                'user_id'      => $data['user_id'],
                'task_degree'  => $data['task_degree'],
                'is_corrected' => true,
                'task_url'     => 'https://example.com',
                'graded_by'    => $instructor->id,
            ]);
        }

        $task->load([
            'user',
            'lecture',
            'gradedBy',
            'feedbacks'
        ]);
        return $this->apiResponseUpdated(new TaskSubmissionResource($task));
    }

    public function gradeSubmissionAll(Request $request, Round $round, Lecture $lecture){

        $instructor = auth()->user();
        $this->ensureTaskBelongsToLecture($round, $lecture, null);

        $data=$request->validate([
            'task_degree' => 'required|numeric',
            'user_id'=>'required|array',
        ]);
        foreach ($request->input('user_id') as $userId) {
            $task = TaskSubmission::where('lecture_id', $lecture->id)
                ->where('user_id', $userId)
                ->first();

            if ($task) {
                $task->update([
                    'task_degree'  => $data['task_degree'],
                    'is_corrected' => true,
                    'graded_by'    => $instructor->id,
                ]);
            } else {
                $task = TaskSubmission::create([
                    'lecture_id'   => $lecture->id,
                    'user_id'      => $userId,
                    'task_degree'  => $data['task_degree'],
                    'is_corrected' => true,
                    'task_url'     => 'https://example.com',
                    'graded_by'    => $instructor->id,
                ]);
            }
        }
        return $this->apiResponseUpdated([]);
    }
    public function getFeedback(Round $round, Lecture $lecture, TaskSubmission $task)
    {
        $this->ensureTaskBelongsToLecture($round, $lecture, $task);

        $task = TaskSubmission::select('id')
            ->where('id', $task->id)
            ->with('feedbacks')
            ->firstOrFail();

        return $this->apiResponseShow(new TaskSubmissionResource($task));
    }

    public function storeFeedback(StoreFeedbackRequest $request, Round $round, Lecture $lecture, TaskSubmission $task)
    {
        $this->ensureTaskBelongsToLecture($round, $lecture, $task);

        $data = $request->validated();

        // Handle image/audio uploads
        if ($data['type'] == FeedbackType::IMAGE->value || $data['type'] == FeedbackType::AUDIO->value) {
            $data['content'] = $this->upload($data['content'], 'tasks/feedback');
        }

        // Create feedback
        $taskFeedback=TaskFeedback::create([
            'task_submission_id' => $task->id,
            'content'            => $data['content'],
            'type'               => $data['type'],
            'note'               => $data['note'] ?? null,
        ]);

        return $this->apiResponseUpdated(new TaskFeedbackResource($taskFeedback));
    }

    public function updateFeedback(StoreFeedbackRequest $request, TaskFeedback $taskFeedback)
    {
        // abort_if($task->lecture_id != $lecture->id, 404);
        $data = $request->validated();
        if($data['type'] == FeedbackType::IMAGE->value || $data['type'] == FeedbackType::AUDIO->value){
            $data['content'] = $this->upload($data['content'],'tasks/feedback');
        }
        $taskFeedback->update([
            'content'            => $data['content'],
            'type'               => $data['type'],
            'note'               => $data['note']??null,
        ]);
        return $this->apiResponseUpdated(new TaskFeedbackResource($taskFeedback));
    }

    public function destroyTaskFeedback(TaskFeedback $taskFeedback)
    {
        $taskFeedback->delete();
        if($taskFeedback->type == FeedbackType::IMAGE->value || $taskFeedback->type == FeedbackType::AUDIO->value){
            MediaHandler::deleteMedia($taskFeedback->content);
        }
        return $this->apiResponseDeleted();
    }

    public function addBonus(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'bonus'   => 'required|integer|min:0|max:100',
        ]);

        $user = User::findOrFail($request->user_id);

        $user->update([
            'bonus' => $request->bonus,
        ]);
        $user->save();

        return $this->apiResponseUpdated([]);
    }


    public function addSkillsStudents(Round $round){
        $course = $round->course;

        foreach($round->students as $student){
            if($this->checkScore($student, $round)){
                $student->skills()->create([
                    'title' => $course->title,
                    'course_id' => $course->id,
                ]);
            }
        }

        return $this->apiResponseUpdated([]);
    }


    public function checkScore(User $user, Round $round){
        $score = min(100, round((($this->getAttendanceRate($round, $user) + $round->getTaskSubmissionRateByUser()) / 2) + $user->bonus, 2));

        return $score >= 80;
    }

    public function getAttendanceRate(Round $round, User $user)
    {
        $lectureIds = $round->lectures()->pluck('lectures.id');

        $totalLectures = $lectureIds->count();

        $actualAttendances = Evaluation::whereIn('lecture_id', $lectureIds)
            ->where('user_id', $user->id)
            ->where('is_attend', true)
            ->count();

        return round(($actualAttendances / $totalLectures) * 100, 2);
    }

    public function getProjects(){
        $projects = Project::orderBy('id','desc')->with('user','acceptedBy')->paginate(50);
        return $this->apiResponseShow(ProjectResource::collection($projects));
    }

    public function acceptProject(Project $project, Request $request){
        $request->validate([
            'accepted' => 'required|boolean',
            'comment' => 'nullable|string',
        ]);
        $project->update([
            'accepted_by' => $request->accepted ? auth()->user()->id : null,
            'accepted' => $request->accepted,
            'comment' => $request->comment,
        ]);
        return $this->apiResponseUpdated([]);
    }

}
