<?php

namespace App\Http\Controllers\Api\V1\Instructor;

use App\Enums\DefineStatus;
use App\Http\Controllers\Controller;
use App\Http\Resources\Lecture\LectureResource;
use App\Http\Resources\Notice\NoticeResource;
use App\Http\Resources\Round\RoundResource;
use App\Http\Resources\User\UserResource;
use App\Http\Traits\EnsuresInstructorAccess;
use App\Http\Traits\MediaHandler;
use App\Models\Lecture;
use App\Models\Notice;
use App\Models\Round;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class InstructorRoundController extends Controller
{
    use MediaHandler;
    use EnsuresInstructorAccess;

    public function getInstructorRounds(): AnonymousResourceCollection
    {
        $instructorId = (int) auth()->id();

        $rounds = Round::select('id', 'name', 'students_count', 'whatsapp_group_link', 'course_id', 'instructor_id', 'status','additional_instructors','link_url','type','round_content')
            ->where('status', DefineStatus::ACTIVE)
            ->with(['course:id,title','notices.createdBy'])
            ->where(function($query) use ($instructorId) {
                $query->where('instructor_id', $instructorId)
                ->orWhereJsonContains('additional_instructors',(string) $instructorId);
            })
            ->useFilters()
            ->latest()
            ->dynamicPaginate(50);

        return RoundResource::collection($rounds);
    }

    public function makeAsRead(Round $round){
        $round->notices()->each(function($notice){
            $notice->read=true;
            $notice->save();
        });
        return $this->apiResponseShow($round);
    }

    public function getRoundsDetails(Round $round)
    {
        $this->ensureRoundAndInstructor($round);

        $lectures = Lecture::with(['zoomMeetings:id,lecture_id,ref_id,start_url,topic,duration,agenda,start_time','lectureTasks','lectureVideos','round.students'])
            ->where('round_id', $round->id)
            ->whereDate('start_time', '<=', now()->toDateString())
            ->orderByDesc('start_time')
            ->paginate(50);

        return LectureResource::collection($lectures);
    }

    public function getRoundStudents(Round $round): AnonymousResourceCollection
    {
        $this->ensureRoundAndInstructor($round);

        $students = $round->studentRounds()
            ->with(['enrolledRounds.lectures'])
            ->select('users.id', 'users.name', 'users.email', 'users.phone', 'users.status', 'round_students.total_points', 'round_students.created_at','round_students.round_id','users.bonus')
            ->orderBy('round_students.created_at', 'DESC')
            ->dynamicPaginate(50);

        return UserResource::collection($students);
    }

    public function editRoundContent(Request $request, Round $round)
    {
        $request->validate([
            'round_content' => 'required|array',
        ]);
        $round->update([
            'round_content' => $request->input('round_content')
        ]);
        return $this->apiResponseUpdated([]);
    }
}
