<?php

namespace App\Http\Controllers\Api\V1\Instructor;

use App\Enums\UserType;
use App\Http\Controllers\Controller;
use App\Http\Requests\Template\SendMessageRequest;
use App\Http\Traits\EnsuresInstructorAccess;
use App\Jobs\SendMessageJob;
use App\Models\Round;
use App\Models\Template;
use App\Models\User;

class InstructorMessageController extends Controller
{
    use EnsuresInstructorAccess;

    public function sendMessage(SendMessageRequest $request, Round $round)
    {
        $this->ensureRoundAndInstructor($round);

        $data = $request->validated();
        $template = Template::with('variables:id,template_id,name')->findOrFail($data['template_id']);

        // Extract required variables from the template
        $requiredVariables = $template->variables->pluck('name')->toArray();
        $providedVariables = $data['variables'];

        // Check if all required variables are provided
        foreach ($requiredVariables as $var) {
            if (!array_key_exists($var, $providedVariables)) {
                return $this->apiResponse(__('validation.custom.valid_variables', ['variable' => $var]), 422);
            }
        }

        // Replace placeholders with provided values
        $message = $template->message;
        foreach ($providedVariables as $key => $value) {
            $message = str_replace("{{{$key}}}", $value, $message);
        }

        $roundStudentIds = $round->roundStudents()->pluck('users.id')->toArray();
        $selectedStudentIds = $data['student_ids'];

         // Ensure all selected students exist in the round
        $invalidStudents = array_diff($selectedStudentIds, $roundStudentIds);
        if (!empty($invalidStudents)) {
            return $this->apiResponse([], __('validation.custom.valid_students'), 422);
        }

        $students = User::select('id', 'name', 'phone')
            ->whereIntegerInRaw('id', $selectedStudentIds)
            ->where('type', UserType::STUDENT->value)
            ->get();

        if ($students->isEmpty()) {
            return $this->apiResponse([], __('validation.custom.valid_students'), 422);
        }

        $sentTo = [];
        foreach ($students as $student) {
            $sentTo[] = [
                'name' => $student->name,
                'phone' => $student->phone,
            ];
        }

        SendMessageJob::dispatch($sentTo, $message)
            ->onQueue('whatsapp-messages')
            ->delay(now()->addSeconds(10));

        return $this->apiResponse([
            'message' => $message,
            'sent_to' => $sentTo,
        ]);
    }
}
