<?php

namespace App\Http\Controllers\Api\V1\Instructor;

use App\Enums\DefineStatus;
use App\Http\Controllers\Controller;
use App\Http\Resources\Home\InstructorHomeResource;
use App\Http\Resources\Lecture\LectureResource;
use App\Models\Lecture;
use App\Models\Round;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;

class InstructorHomeController extends Controller
{
    public function showHomeStatistics()
    {
        $instructorId = auth()->id();

        $statistics = [
            'total_courses'  => DB::table('courses')
                ->whereIn('id', function ($query) use ($instructorId) {
                    $query->select('course_id')
                        ->from('rounds')
                        ->where(function($query) use($instructorId){
            $query->where('instructor_id', $instructorId)
                ->orWhereJsonContains('additional_instructors', (string) $instructorId);
        });
                })
                ->count() ?? 0,

            'total_rounds'   => DB::table('rounds')
                ->where(function($query) use($instructorId){
            $query->where('instructor_id', $instructorId)
                ->orWhereJsonContains('additional_instructors', (string) $instructorId);
        })
                ->count() ?? 0,

            'active_rounds'  => DB::table('rounds')
                ->where(function($query) use($instructorId){
            $query->where('instructor_id', $instructorId)
                ->orWhereJsonContains('additional_instructors', (string) $instructorId);
        })
                ->where('status', DefineStatus::ACTIVE)
                ->count() ?? 0,

            'total_students' => DB::table('round_students')
                ->whereIn('round_id', function ($query) use ($instructorId) {
                    $query->select('id')
                        ->from('rounds')
                        ->where(function($query) use($instructorId){
            $query->where('instructor_id', $instructorId)
                ->orWhereJsonContains('additional_instructors', (string) $instructorId);
        });
                })
                ->distinct('user_id')
                ->count() ?? 0,

                'total_lectures' => DB::table('lectures')
                    ->whereIn('round_id', function ($query) use ($instructorId) {
                        $query->select('id')
                            ->from('rounds')
                            ->where(function($query) use($instructorId){
            $query->where('instructor_id', $instructorId)
                ->orWhereJsonContains('additional_instructors', (string) $instructorId);
        });
                    })
                    ->count() ?? 0,

                'total_graded_tasks' => DB::table('task_submissions')
                    ->where('graded_by', $instructorId) // Only tasks graded by the auth instructor
                    ->whereIn('lecture_id', function ($query) use ($instructorId) {
                        $query->select('id')
                            ->from('lectures')
                            ->whereIn('round_id', function ($subQuery) use ($instructorId) {
                                $subQuery->select('id')
                                    ->from('rounds')
                                    ->where(function($query) use($instructorId){
            $query->where('instructor_id', $instructorId)
                ->orWhereJsonContains('additional_instructors', (string) $instructorId);
        });
                            });
                    })
                    ->count() ?? 0,
        ];

        return $this->apiResponseShow(new InstructorHomeResource($statistics));
    }

    public function instructorUpcomingMeetings()
    {
        $instructorId = auth()->id();
        $today = now()->toDateString();
        $nextWeek = now()->addDays(7)->toDateString();

        $roundIds = Round::where(function($query) use($instructorId){
            $query->where('instructor_id', $instructorId)
                ->orWhereJsonContains('additional_instructors', (string) $instructorId);
        })
            ->where('status', DefineStatus::ACTIVE)
            ->pluck('id');

        $lectures = Lecture::query()
            ->select('lectures.*')
            ->from('lectures')
            ->join('zoom_meetings', 'lectures.id', '=', 'zoom_meetings.lecture_id')
            ->whereIn('lectures.round_id', $roundIds)
            ->whereBetween('zoom_meetings.start_time', [$today, $nextWeek])
            ->active()
            ->with([
                'round:id,name',
                'zoomMeetings' => function ($query) use ($today, $nextWeek) {
                    $query->whereBetween('start_time', [$today, $nextWeek]);
                },
                'lectureVideos'
            ])
            ->get();

        return LectureResource::collection($lectures);
    }

    public function getTopStudentPerRound(): JsonResponse
    {
        $instructor = auth()->user();

        [$rounds, $allStudents] = $this->getRoundsAndStudents($instructor);

        $topStudents = $this->calculateTopStudents($rounds, $allStudents);

        return $this->apiResponseShow($topStudents);
    }

    /**
     * Fetch rounds the instructor is assigned to and all students in those rounds.
     *
     * @param \App\Models\User $instructor
     * @return array
     */
    private function getRoundsAndStudents($instructor)
    {
        $rounds = Round::where(function($query) use($instructor){
            $query->where('instructor_id', $instructor->id)
                ->orWhereJsonContains('additional_instructors', (string) $instructor->id);
        })
            ->with('lectures')
            ->get();

        // Collect lecture IDs from all rounds
        $lectureIds = $rounds->flatMap->lectures->pluck('id')->unique();

        // Get all students enrolled in these rounds
        $allStudents = User::whereHas('enrolledRounds', function ($query) use ($rounds) {
            $query->whereIntegerInRaw('round_id', $rounds->pluck('id'));
        })->with([
            'taskSubmissions' => function ($query) use ($lectureIds) {
                $query->whereIntegerInRaw('lecture_id', $lectureIds);
            },
            'evaluations' => function ($query) use ($lectureIds) {
                $query->whereIntegerInRaw('lecture_id', $lectureIds);
            },
            'enrolledRounds'
        ])->get();

        return [$rounds, $allStudents];
    }

    /**
     * Calculate the top student for each round based on total points.
     *
     * @param \Illuminate\Support\Collection $rounds
     * @param \Illuminate\Support\Collection $allStudents
     * @return \Illuminate\Support\Collection
     */
    private function calculateTopStudents($rounds, $allStudents)
    {
        return $rounds->map(function ($round) use ($allStudents) {
            // Get students in this specific round
            $students = $allStudents->filter(fn($student) =>
                $student->enrolledRounds->contains('id', $round->id)
            );

            $roundLectureIds = $round->lectures->pluck('id');

            // Calculate scores
            $studentScores = $students->map(function ($student) use ($roundLectureIds) {
                $attendancePoints = $student->evaluations
                    ->whereIn('lecture_id', $roundLectureIds)
                    ->sum('attendance_degree');

                $taskDegree = $student->taskSubmissions
                    ->whereIn('lecture_id', $roundLectureIds)
                    ->sum('task_degree');

                return [
                    'student_id'   => $student->id,
                    'name'         => $student->name,
                    'total_points' => $attendancePoints + $taskDegree,
                ];
            });

            if ($studentScores->isEmpty() || $studentScores->sum('total_points') === 0) {
                return null;
            }

            return [
                'round_id'    => $round->id,
                'round_name'  => $round->name,
                'top_student' => $studentScores->sortByDesc('total_points')->first() ?? null,
            ];
        })->filter()->values();
    }
}
