<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\InstallmentStatusEnum;
use App\Enums\PaymentMethodEnum;
use App\Enums\PaymentStatus;
use App\Enums\TransactionTypeEnum;
use App\Http\Controllers\Controller;
use App\Http\Resources\Installment\InstallmentResource;
use App\Models\CourseEnrollment;
use App\Models\Installment;
use App\Models\Payment;
use App\Models\Round;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\In;

class InstallmentController extends Controller
{
    public function index()
    {
        $installments=Installment::with(['user', 'courseEnrollment', 'transactions'])->useFilters()->dynamicPaginate(50);
        return InstallmentResource::collection($installments);
    }

    public function getInstallmentsByRound($round_id)
    {
        $round=Round::findOrFail($round_id);
        $ids=$round->courseEnrollments()->pluck('id')->toArray();
        $installments=Installment::whereIn('course_enrollment_id',$ids)->with(['user', 'courseEnrollment', 'transactions'])->useFilters()->dynamicPaginate(50);
        return InstallmentResource::collection($installments);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'user_id' => 'required|exists:users,id',
            'course_enrollment_id' => 'required|exists:course_enrollments,id',
            'amount' => 'required|numeric',
            'due_date' => 'required|date',
            'paid_at' => 'nullable|date',
            'status' => ['nullable', Rule::enum(InstallmentStatusEnum::class)],
            'notes' => 'nullable|string',
        ]);

        $installment = Installment::create($data);

        return $this->apiResponseStored(new InstallmentResource($installment->load(['user', 'courseEnrollment'])));
    }

    public function show(Installment $installment)
    {
        return new InstallmentResource($installment->load(['user', 'courseEnrollment']));
    }

    public function update(Request $request, Installment $installment)
    {
        $data = $request->validate([
            'amount' => 'sometimes|numeric',
            'due_date' => 'sometimes|date',
            'paid_at' => 'nullable|date',
            'status' => ['nullable', Rule::enum(InstallmentStatusEnum::class)],
            'notes' => 'nullable|string',
        ]);

        $installment->update($data);

        if ($installment->status === InstallmentStatusEnum::PAID->value) {
            Transaction::create([
                'user_id'         => $installment->user_id,
                'course_enrollment_id'  => $installment->course_enrollment_id,
                'installment_id'  => $installment->id,
                'amount'          => $installment->amount,
                'payment_method'  => PaymentMethodEnum::CASH,
                'payment_date'    => now(),
                'type'            => TransactionTypeEnum::INCOME,
                'notes'           => 'Auto transaction from installment payment',
                'payment_method_id' => $request->input('payment_method_id'),
                'accepted_by'     => Auth::id(),
                'created_by'      => Auth::id(),
                'updated_by'      => Auth::id(),
            ]);
        }

        return $this->apiResponseUpdated(new InstallmentResource($installment->load(['user', 'courseEnrollment'])));
    }

    public function destroy(Installment $installment)
    {
        $installment->transactions()->delete();
        $installment->delete();

        return $this->apiResponseDeleted();
    }

    public function changeStatus(Request $request, Installment $installment)
    {
        $request->validate([
            'status' => ['required', Rule::enum(InstallmentStatusEnum::class)],
        ]);

        $newStatus = $request->input('status');

        $installment->update([
            'status' => $newStatus,
            'paid_at' => $newStatus == InstallmentStatusEnum::PAID->value ? now() : null,
        ]);

        $courseEnrollment = CourseEnrollment::find($installment->course_enrollment_id);

        if ($newStatus == InstallmentStatusEnum::PAID->value) {
            Transaction::create([
                'user_id'              => $installment->user_id,
                'course_enrollment_id' => $installment->course_enrollment_id,
                'installment_id'       => $installment->id,
                'amount'               => $installment->amount,
                'payment_method'       => PaymentMethodEnum::CASH,
                'payment_date'         => now(),
                'type'                 => TransactionTypeEnum::INCOME,
                'notes'                => 'Auto transaction from installment payment',
                'payment_method_id'    => $request->input('payment_method_id'),
                'accepted_by'          => Auth::id(),
                'created_by'           => Auth::id(),
                'updated_by'           => Auth::id(),
            ]);
        }

        if ($courseEnrollment->installments()->where('status', '!=', InstallmentStatusEnum::PAID->value)->doesntExist()) {
            $courseEnrollment->update([
                'payment_status' => PaymentStatus::PAID,
            ]);
        }

        return $this->apiResponseUpdated([
            'message'     => 'Installment status updated successfully.',
            'installment' => $installment,
        ]);
    }

}
