<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\DefineStatus;
use App\Http\Controllers\Controller;
use App\Http\Resources\Course\CourseResource;
use App\Http\Resources\Home\HomeResource;
use App\Http\Traits\ApiResponse;
use App\Models\Course;
use App\Models\Round;
use App\Models\User;
use App\Services\ReportService;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class HomeController extends Controller
{
    use ApiResponse;
    private ReportService $reportService;

    /**
     * Inject the ReportService dependency.
     *
     * @param ReportService $reportService
     */
    public function __construct(ReportService $reportService)
    {
        $this->reportService = $reportService;
    }

    /**
     * Show summary statistics for the home dashboard.
     *
     * @return JsonResponse
     */
    public function showHomeStatistics(): JsonResponse
    {
        try {
            $statistics = [
                'logged_in_users' => $this->safeCount(User::where('is_open', 1)),
                'total_courses'       => $this->safeCount(DB::table('courses')),
                'active_courses'      => $this->safeCount(DB::table('courses')->where('status', DefineStatus::ACTIVE)),
                'total_rounds'        => $this->safeCount(Round::query()),
                'active_rounds'       => $this->safeCount(Round::where('status', DefineStatus::ACTIVE)),
                'total_students'      => $this->safeCount(User::students()),
                'active_students'     => $this->safeCount(User::students()->where('status', DefineStatus::ACTIVE)),
                'total_supervisors'   => $this->safeCount(User::supervisors()),
                'active_supervisors'  => $this->safeCount(User::supervisors()->where('status', DefineStatus::ACTIVE)),
            ];
            return $this->apiResponseShow(new HomeResource($statistics));
        } catch (\Exception $e) {
            // Return default statistics if database is empty or tables don't exist
            $statistics = [
                'logged_in_users' => 0,
                'total_courses'       => 0,
                'active_courses'      => 0,
                'total_rounds'        => 0,
                'active_rounds'       => 0,
                'total_students'      => 0,
                'active_students'     => 0,
                'total_supervisors'   => 0,
                'active_supervisors'  => 0,
            ];
            return $this->apiResponseShow(new HomeResource($statistics));
        }
    }

    /**
     * Show distribution of users by type.
     *
     * @return JsonResponse
     */
    public function ShowUserDistribution(): JsonResponse
    {
        try {
            $UserDistribution = [
                'total_students'      => $this->safeCount(User::students()),
                'total_instructors'   => $this->safeCount(User::instructors()),
                'total_supervisors'   => $this->safeCount(User::supervisors()),
                'total_admins'        => $this->safeCount(User::admins()),
            ];
            return $this->apiResponseShow(new HomeResource($UserDistribution));
        } catch (\Exception $e) {
            // Return default distribution if database is empty or tables don't exist
            $UserDistribution = [
                'total_students'      => 0,
                'total_instructors'   => 0,
                'total_supervisors'   => 0,
                'total_admins'        => 0,
            ];
            return $this->apiResponseShow(new HomeResource($UserDistribution));
        }
    }

    public function recentData()
    {
        try {
            $courses = Course::query()
                ->latest()
                ->limit(10)
                ->get();

            $rounds = Round::query()
                ->latest()
                ->limit(10)
                ->get();

            return $this->apiResponseShow(new HomeResource([
                'recent_courses' => $courses,
                'recent_rounds'  => $rounds,
            ]));
        } catch (\Exception $e) {
            // Return empty collections if database is empty or tables don't exist
            return $this->apiResponseShow(new HomeResource([
                'recent_courses' => collect([]),
                'recent_rounds'  => collect([]),
            ]));
        }
    }

    public function topSellingCourses()
    {
        try {
            $topSelling = Course::query()->whereHas('enrollments', function($query){
                    $query->where('status', DefineStatus::ACTIVE);
                })
                ->latest()
                ->limit(5)
                ->get();

            return $this->apiResponseShow(CourseResource::collection($topSelling));
        } catch (\Exception $e) {
            // Return empty collection if database is empty or tables don't exist
            return $this->apiResponseShow(CourseResource::collection(collect([])));
        }
    }

    public function topSellingCoursesReport(Request $request): JsonResponse
    {
        $filters = $request->only(['year']);

        $query = Course::query()->whereHas('enrollments', function($query){
            $query->where('status', DefineStatus::ACTIVE);
        })
        ->latest()
        ->limit(5);
        $report = $this->reportService->generateReport($query, $filters);

        return $this->apiResponse($report, __('main.success_query'), 200);
    }

    /**
     * Show a detailed student report based on the applied filters.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function ShowStudentReport(Request $request): JsonResponse
    {
        $filters = $request->only(['year']);

        $query = User::students();

        // Generate the report using the ReportService with the query and filters.
        $report = $this->reportService->generateReport($query, $filters);

        return $this->apiResponse($report, __('main.success_query'), 200);
    }

    /**
     * Safely count records from a query builder, returning 0 if the query fails.
     *
     * @param mixed $query
     * @return int
     */
    private function safeCount($query): int
    {
        try {
            return $query->count();
        } catch (\Exception $e) {
            return 0;
        }
    }
}
