<?php

namespace App\Http\Controllers\Api\V1;

use App\Models\User;
use App\Models\Round;
use App\Enums\UserType;
use App\Models\Lecture;
use App\Models\Evaluation;
use App\Models\RoundStudent;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use App\Http\Resources\User\UserResource;
use App\Http\Resources\Evaluation\EvaluationResource;
use App\Http\Requests\Evaluation\UpdateEvaluationRequest;
use App\Http\Requests\Evaluation\StudentAttendanceRequest;
use App\Http\Requests\Evaluation\StoreStudentPointsRequest;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class EvaluationController extends Controller
{
    /**
     * Get student evaluations for a specific lecture in a round.
     *
     * Filters students by their evaluations for the lecture and paginates the results.
     *
     * @param  Round  $round
     * @param  Lecture  $lecture
     * @return AnonymousResourceCollection
     */
    public function getEvaluationByLecture(Round $round, Lecture $lecture): AnonymousResourceCollection
    {
        $studentIds = $round->students->pluck('id')->toArray();
            $students = User::whereIn('id', $studentIds)->with(['evaluation' => function ($query) use ($lecture) {
                $query->where('lecture_id', $lecture->id);
            },'taskSubmissions'=> function($q) use($lecture) {
                $q->where('lecture_id', $lecture->id);
            },'absencesNotices'=> function($q) use($lecture) {
                $q->where('lecture_id', $lecture->id);
            }])->dynamicPaginate(40);

        return UserResource::collection($students);
    }


    public function storeStudentAttendance(StudentAttendanceRequest $request, Round $round, Lecture $lecture): JsonResponse
    {
        $lecture->load('round.students');
        // Ensure the lecture belongs to the given round
        abort_if($lecture->round_id != $round->id, 404);

        // Get students in the round
        $studentsInRound = $lecture->round->students->where('type', UserType::STUDENT->value)->pluck('id')->toArray();

        // Filter valid student IDs from the request
        $validStudentIds = array_intersect($request->student_ids, $studentsInRound);

        // Create evaluations
        $evaluations = collect($studentsInRound)->map(function ($student_id) use ($lecture, $validStudentIds) {
            return [
                'lecture_id' => $lecture->id,
                'user_id'    => $student_id,
                'is_attend'  => in_array($student_id, $validStudentIds),
                'created_at' => now(),
                'updated_at' => now(),
            ];
        })->toArray();

        Evaluation::upsert(
            $evaluations,
            ['lecture_id', 'user_id'],
            ['is_attend', 'updated_at']
        );

        // $evaluatedStudents = User::whereIn('id', $studentsInRound)
        //     ->with('evaluation.lecture')
        //     ->get();

        // Return the evaluated students
        $evaluatedStudents = User::whereIn('id', $studentsInRound)
            ->with(['evaluation.lecture' => function ($query) {
                $query->with('evaluations'); // Eager load evaluations on the lecture
            }])
            ->get();

        return $this->apiResponseStored(UserResource::collection($evaluatedStudents));
    }

    /**
     * Store or update a student's evaluation for a specific lecture.
     *
     * Calculates the overall rate and updates the evaluation record.
     *
     * @param  UpdateEvaluationRequest  $request
     * @param  Round  $round
     * @param  Lecture  $lecture
     * @return \Illuminate\Http\JsonResponse
     */
    public function storeStudentEvaluation(UpdateEvaluationRequest $request, $round, Lecture $lecture): JsonResponse
    {
        $data = $request->validated();

        $lecture->load(['round.students']);

        // Verify student is in the round
        abort_if(!$lecture->round->students->contains($data['user_id']), 404);

        // Get current evaluation
        $currentEvaluation = Evaluation::where('lecture_id', $lecture->id)
            ->where('user_id', $data['user_id'])
            ->first();

        // Ensure oldAttendanceDegree is always a float
        $oldAttendanceDegree = $currentEvaluation?->assignment_degree ?? 0.0;

        // Update evaluation
        $evaluation = Evaluation::updateOrCreate(
            [
                'lecture_id' => $lecture->id,
                'user_id' => $data['user_id']
            ],
            [
                'assignment_degree' => $data['assignment_degree']
            ]
        );

        // Update round student points
        $this->updateRoundStudentPoints(
            $lecture->round->id,
            $data['user_id'],
            (float) $data['assignment_degree'],
            (float) $oldAttendanceDegree
        );

        return $this->apiResponseStored(EvaluationResource::make($evaluation));
    }


    /**
     * Update the total points for a student in a round based on attendance degree change
     */
    private function updateRoundStudentPoints(int $roundId, int $userId, float $newDegree, float $oldDegree): void
    {
        $roundStudent = RoundStudent::where('round_id', $roundId)
            ->where('user_id', $userId)
            ->first();

        $pointsDifference = $newDegree - $oldDegree;

        if ($pointsDifference > 0) {
            $roundStudent->increment('total_points', $pointsDifference);
        } elseif ($pointsDifference < 0) {
            $roundStudent->decrement('total_points', abs($pointsDifference));
        }
    }



    /**
     * Store points for a specific student in a round.
     *
     * Validates the data and creates an evaluation record with the points.
     *
     * @param  StoreStudentPointsRequest  $request
     * @param  Round  $round
     * @param  User  $student
     * @return \Illuminate\Http\JsonResponse
     */
    public function storeStudentPoints(StoreStudentPointsRequest $request, Round $round, User $student): JsonResponse
    {
        $data = $request->validated();
        $round->load('students');
        abort_if(!$round->students->contains($student), 404);
        $roundStudent = RoundStudent::where('round_id', $round->id)->where('user_id', $student->id)->first();
        $roundStudent->increment('total_points', $data['points']);
        return $this->apiResponseUpdated([]);
    }
}
