<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\InstallmentStatusEnum;
use App\Enums\PaymentMethodEnum;
use App\Enums\PaymentStatus;
use App\Enums\PaymentTypeEnum;
use App\Enums\TransactionTypeEnum;
use App\Http\Requests\CourseEnrollment\AdminCreateCourseEnrollmentRequest;
use App\Models\CourseEnrollment;
use App\Http\Controllers\Controller;
use App\Http\Requests\CourseEnrollment\CreateCourseEnrollmentRequest;
use App\Http\Resources\CourseEnrollment\CourseEnrollmentResource;
use App\Models\Installment;
use App\Models\Round;
use App\Models\Transaction;
use App\Notifications\InvoicePaidMail;
use App\Services\GenerateInvoice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Str;
use Illuminate\Validation\Rule;

class CourseEnrollmentController extends Controller
{
    // public function index()
    // {
    //     $enrollments = CourseEnrollment::with(['user', 'course'])
    //         ->useFilters()
    //         ->get();
    //     $filtered = $enrollments->filter(function ($enrollment) {
    //         $courseId = $enrollment->course_id;
    //         $userId = $enrollment->user_id;
    //         $roundsIds = \App\Models\Round::where('course_id', $courseId)->pluck('id');
    //         $isUserInGroup = \App\Models\RoundStudent::where('user_id', $userId)
    //             ->whereIn('round_id', $roundsIds)
    //             ->exists();

    //         return $isUserInGroup;
    //     });

    //     return CourseEnrollmentResource::collection($filtered);
    // }

    public function index()
    {
        $enrollments = CourseEnrollment::with(['user', 'course'])
            ->whereDoesntHave('course.rounds.students', function ($query) {
                $query->whereColumn('round_students.user_id', 'course_enrollments.user_id');
            })
            ->useFilters()
            ->dynamicPaginate(50);

        return CourseEnrollmentResource::collection($enrollments);
    }


    public function store(AdminCreateCourseEnrollmentRequest $request)
    {
        $data = $request->validated();

        $courseEnrollment = CourseEnrollment::create(array_merge($data, [
            'payment_method' => 'cash',
            'payment_status' => true,
            'status' => true,
            'transaction_id' => $this->generateTransactionId(),
            'created_by'=>auth()->user()->id
        ]));

        try {
            $invoicePath = $this->generateCourseInvoice($courseEnrollment);

            $this->sendEnrollmentNotification($courseEnrollment, $invoicePath);

        } catch (\Exception $e) {
            Log::error('Failed to generate or send invoice notification.', [
                'enrollment_id' => $courseEnrollment->id,
                'error' => $e->getMessage(),
                'stack_trace' => $e->getTraceAsString(),
            ]);

        }

        return $this->apiResponseStored(new CourseEnrollmentResource($courseEnrollment));
    }

    protected function generateTransactionId()
    {
        return Str::random(10);
    }

    protected function sendEnrollmentNotification($enrollment, $invoicePath)
    {
        if (!$enrollment->user || !$enrollment->course) {
            throw new \Exception('Missing required relationship data for notification');
        }

        $notificationData = [
            'title' => 'Your Course Payment Receipt',
            'message' => 'Your payment for ' . $enrollment->course->title . ' has been processed successfully.',
            'subject' => 'Payment Confirmation for ' . $enrollment->course->title,
        ];

        Notification::sendNow($enrollment->user, new InvoicePaidMail($invoicePath, $notificationData));
    }

    protected function generateCourseInvoice(CourseEnrollment $enrollment): string
    {
        if (!$enrollment->course || !$enrollment->user) {
            throw new \Exception('Missing required relationship data for invoice');
        }

        $invoiceService = new GenerateInvoice();

        try {
            return $invoiceService
                ->setCustomer([
                    'name' => $enrollment->user->name,
                    'address' => $enrollment->user->address ?? 'N/A',
                    'email' => $enrollment->user->email,
                    'phone' => $enrollment->user->phone ?? 'N/A',
                ])
                ->setSeller([
                    'name' => config('app.name', 'Course Provider'),
                    'address' => config('app.address', 'N/A'),
                    'email' => config('app.email', 'contact@example.com'),
                    'phone' => config('app.phone', 'N/A'),
                ])
                ->addItem(
                    $enrollment->course->title,
                    $enrollment->total_amount,
                    1
                )
                ->setLogo(config('app.logo'))
                ->setInvoiceNumber($enrollment->id)
                ->saveToDisk('public', 'invoices')
                ->save();
        } catch (\Exception $e) {
            Log::error('Invoice generation failed', ['error' => $e->getMessage()]);
            throw $e;
        }
    }


    public function changeStatus(Request $request, CourseEnrollment $enrollment)
    {
        $request->validate([
            'status' => 'required',
        ]);
        $enrollment->payment_status = $request->status;
        $enrollment->status = $request->status;
        $enrollment->save();
        return $this->apiResponseUpdated(new CourseEnrollmentResource($enrollment));
    }


    public function updateEnrollment(Request $request, CourseEnrollment $courseEnrollment)
    {
        $data = $request->validate([
            'course_id' => 'sometimes|exists:courses,id',
            'user_id' => 'sometimes|exists:users,id',
            'total_amount' => 'sometimes|numeric',
            'enrollment_date' => 'sometimes|date',
            'status' => 'sometimes|boolean',
            'payment_method' => 'nullable|string',
            'payment_status' => ['nullable', Rule::enum(PaymentStatus::class)],
            'transaction_id' => 'nullable|uuid',
            'coupon_id' => 'nullable|exists:coupons,id',
            'round_id' => 'nullable|exists:rounds,id',
            'payment_type' => ['nullable', Rule::enum(PaymentTypeEnum::class)],
            'coupon_amount' => 'nullable|numeric',
            'installments' => 'nullable|array',
            'installments.*.id'=>'nullable',
            'installments.*.amount' => 'numeric',
            'installments.*.due_date' => 'date'
        ]);

        if (isset($data['payment_type'])) {
            $data['payment_status'] = match ($data['payment_type']) {
                PaymentTypeEnum::CASH->value => PaymentStatus::PAID->value,
                PaymentTypeEnum::INSTALLMENT->value => PaymentStatus::IN_PROGRESS->value,
                default => $data['payment_status'] ?? null,
            };
        }


        $courseEnrollment->update($data);

        if (
            $courseEnrollment->payment_type === PaymentTypeEnum::INSTALLMENT->value &&
            !empty($data['installments'])
        ) {
            foreach ($data['installments'] as $installment) {
                $payload = [
                    'user_id'               => $courseEnrollment->user_id,
                    'course_enrollment_id'  => $courseEnrollment->id,
                    'amount'                => $installment['amount'],
                    'due_date'              => $installment['due_date'],
                    'paid_at'               => null,
                    'status'                => InstallmentStatusEnum::PENDING->value,
                    'notes'                 => 'Auto transaction from installment payment',
                ];
                if (!empty($installment['id'])) {
                    $model = Installment::findOrFail($installment['id']);
                    $model->fill($payload);
                    $model->updated_by = Auth::id();
                    $model->save();
                } else {
                    Installment::create($payload + [
                        'created_by' => Auth::id(),
                        'updated_by' => Auth::id(),
                    ]);
                }
            }
        }

        if (
            $courseEnrollment->payment_type === PaymentTypeEnum::CASH->value &&
            $courseEnrollment->payment_status === PaymentStatus::PAID->value &&
            $request->filled('payment_method_id')
        ) {
            Transaction::create([
                'user_id'         => $courseEnrollment->user_id,
                'course_enrollment_id' => $courseEnrollment->id,
                'amount'          =>($courseEnrollment->total_amount-$courseEnrollment->coupon_amount),
                'payment_method'  => PaymentMethodEnum::CASH,
                'payment_date'    => now(),
                'type'            => TransactionTypeEnum::INCOME,
                'notes'           => 'Auto transaction from cash payment',
                'payment_method_id' => $request->input('payment_method_id'),
                'accepted_by'     => Auth::id(),
                'created_by'      => Auth::id(),
                'updated_by'      => Auth::id(),
            ]);
        }


        return $this->apiResponseUpdated(new CourseEnrollmentResource($courseEnrollment->load('transactions')));
    }


    public function getCourseEnrollmentsByRound(Round $round)
    {
        $courseEnrollments = CourseEnrollment::where('round_id', $round->id)->with('user', 'course','installments')->dynamicPaginate(50);
        return CourseEnrollmentResource::collection($courseEnrollments);
    }

    public function destroy(CourseEnrollment $courseEnrollment)
    {
        if ($courseEnrollment->transactions()->exists()) {
            return $this->apiResponse([
                'message' => 'This enrollment has transactions and cannot be deleted.',
                'status' => false,
            ], 400);
        }

        if ($courseEnrollment->installments()->exists()) {
            return $this->apiResponse([
                'message' => 'This enrollment has installments and cannot be deleted.',
                'status' => false,
            ], 400);
        }

        $courseEnrollment->delete();

        return $this->apiResponseDeleted();
    }

}
