<?php

namespace App\Http\Controllers\Api\V1;

use App\Helpers\ValidationMessageHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\Course\UpdateCourseRequest;
use App\Http\Requests\Course\CreateCourseRequest;
use App\Http\Resources\Course\CourseResource;
use App\Http\Traits\MediaHandler;
use App\Http\Traits\NotificationTrait;
use App\Models\Course;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class CourseController extends Controller
{
    use MediaHandler, NotificationTrait;

    public function index(): AnonymousResourceCollection
    {
        $courses = Course::query()
            ->with('category:id,name','topic')
            ->useFilters()
            ->latest()
            ->dynamicPaginate();
        return CourseResource::collection($courses);
    }

    public function store(CreateCourseRequest $request): JsonResponse
    {
        $data = $request->validated();
        $data['image'] = $request->hasFile('image')
            ? MediaHandler::upload($request->file('image'), 'courses/images')
            : null;
        $data['preview_video_thumbnail'] = $request->hasFile('preview_video_thumbnail')
            ? MediaHandler::upload($request->file('preview_video_thumbnail'), 'courses/video_thumbnails')
            : null;
        $data['learning_plan'] = $request->hasFile('learning_plan')
            ? MediaHandler::upload($request->file('learning_plan'), 'courses/learning_plans')
            : null;
        $course = Course::create($data);
        $course->load('category:id,name','topic');
        // $this->sendPushNotificationClientSide('admin','add_course',$course->category_id);
        return $this->apiResponseStored(new CourseResource($course));
    }

    public function show(Course $course): JsonResponse
    {
        $course->load('category:id,name');
        return $this->apiResponseShow(new CourseResource($course));
    }

    public function update(UpdateCourseRequest $request, Course $course): JsonResponse
    {
        $data = $request->validated();
        if(isset($data['image'])){
            $data['image'] = MediaHandler::updateMedia($data['image'], 'courses/images', $course->image);
        }
        if(isset($data['preview_video_thumbnail'])){
            $data['preview_video_thumbnail'] = MediaHandler::updateMedia($data['preview_video_thumbnail'], 'courses/video_thumbnails', $course->preview_video_thumbnail);
        }
        if(isset($data['learning_plan'])){
            $data['learning_plan'] = MediaHandler::updateMedia($data['learning_plan'], 'courses/learning_plans', $course->learning_plan);
        }
        if(isset($data['status'])){
            $data['status'] = (int) $data['status'];
        }
        $course->update($data);
        $course->load('category:id,name','topic');
        $course->refresh();
        return $this->apiResponseUpdated(new CourseResource($course));
    }

    public function destroy(Course $course): JsonResponse
    {
        if($course->enrollments()->count() > 0){
            return $this->apiResponse([], ValidationMessageHelper::cannotDelete('course', 'enrollments'), 422);
        }
        if($course->rounds()->count() > 0){
            return $this->apiResponse([], ValidationMessageHelper::cannotDelete('course', 'groups'), 422);
        }
        $course->image ? MediaHandler::deleteMedia($course->image) : null;
        $course->preview_video_thumbnail ? MediaHandler::deleteMedia($course->preview_video_thumbnail) : null;
        $course->learning_plan ? MediaHandler::deleteMedia($course->learning_plan) : null;

        $course->delete();
        return $this->apiResponseDeleted();
    }


    public function storeCourseContent(Request $request, Course $course){
        $request->validate([
            'course_content'=>'required|array'
        ]);
        $course->course_content = $request->input('course_content');
        $course->save();
        return $this->apiResponseUpdated(new CourseResource($course));
    }


    public function addSkills(Request $request, Course $course){
        $request->validate([
            'skills'=>'required|array'
        ]);
        $course->skills = $request->input('skills');
        $course->save();
        return $this->apiResponseUpdated(new CourseResource($course));
    }

}
