<?php

namespace App\Http\Controllers\Api\V1\ClientSide\Payment;

use App\Enums\PaymentMethodEnum;
use App\Enums\PaymentStatus;
use App\Models\Course;
use App\Models\Payment;
use App\Services\GenerateInvoice;
use Illuminate\Http\Request;
use App\Models\CourseEnrollment;
use App\Http\Controllers\Controller;
use App\Notifications\InvoicePaidMail;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    public function verifyWithPaymob()
    {
        $data = request()->json()->all();
        if (!empty($data['transaction']['success']) && $data['transaction']['success'] == true) {
            $orderId = $data['transaction']['order']['id'] ?? null;
            if (!$orderId) {
                return $this->apiResponse([], __('main.payment_failed'), 400);
            }
            $courseEnrollment = CourseEnrollment::where('transaction_id', $orderId)->first();
            if ($courseEnrollment) {
                $courseEnrollment->update([
                    'payment_status' => PaymentStatus::PAID->value,
                    'payment_type'=>PaymentMethodEnum::PAYMOB->value,
                    'status'         => true
                ]);
                try {
                    $invoicePath = $this->generateCourseInvoice($courseEnrollment, $data);

                    $notificationData = [
                        'title'   => 'Your Course Payment Receipt',
                        'message' => 'Your payment for ' . $courseEnrollment->course->title . ' has been processed successfully.',
                        'subject' => 'Payment Confirmation for ' . $courseEnrollment->course->title,
                    ];

                    auth()->user()->notify(new InvoicePaidMail($invoicePath, $notificationData));
                } catch (\Exception $e) {
                    Log::error('Failed to generate or send invoice notification.', [
                        'user_id'     => auth()->id(),
                        'course_id'   => $courseEnrollment->course->id ?? null,
                        'error'       => $e->getMessage(),
                        'stack_trace' => $e->getTraceAsString(),
                    ]);
                }
            }
        }
    }

    public function handlePaymentRedirect(Request $request)
    {
        $data = request()->json()->all();
        if (!empty($data['transaction']['success']) && $data['transaction']['success'] == true) {
            $orderId = $data['transaction']['order']['id'] ?? null;
            if (!$orderId) {
                return view('payment.failure');
            }
            $courseEnrollment = CourseEnrollment::where('transaction_id', $orderId)->first();
            if ($courseEnrollment) {
                $course = Course::find($courseEnrollment->course_id);
                return view('payment.success', ['courseTitle' => $course->title]);
            }
        }
        $paymentResponseData = $request->all();
        if ($paymentResponseData['success'] ?? false) {
            $courseEnrollment = CourseEnrollment::where('transaction_id', $paymentResponseData['order_id'] ?? '')->first();
            if ($courseEnrollment) {
                $course = Course::find($courseEnrollment->course_id);
                Payment::create($this->handlePaymentData($paymentResponseData,'online_course_enrollment'));
                try {
                    $invoicePath = $this->generateCourseInvoice($courseEnrollment, $data);

                    $notificationData = [
                        'title'   => 'Your Course Payment Receipt',
                        'message' => 'Your payment for ' . $course->title . ' has been processed successfully.',
                        'subject' => 'Payment Confirmation for ' . $course->title,
                    ];

                    auth()->user()->notify(new InvoicePaidMail($invoicePath, $notificationData));
                } catch (\Exception $e) {
                    Log::error('Failed to generate or send invoice notification.', [
                        'user_id'     => auth()->id(),
                        'course_id'   => $courseEnrollment->course->id ?? null,
                        'error'       => $e->getMessage(),
                        'stack_trace' => $e->getTraceAsString(),
                    ]);
                }
                return view('payment.success', ['courseTitle' => $course->title]);
            }
            return view('payment.success');
        } else {
            $errorMessage = $paymentResponseData['message'] ?? 'Payment failed';
            return view('payment.failure', ['errorMessage' => $errorMessage]);
        }
    }

    protected function handlePaymentData(array $data,string $paymentFor){
        return [
            'payment_for'            => $paymentFor,
            'transaction_id'         => $data['transaction']['id'] ?? null,
            'order_id'               => $data['transaction']['order']['id'] ?? null,
            'amount'                 => $data['transaction']['amount_cents'] ?? 0,
            'currency'               => $data['transaction']['currency'] ?? 'EGP',
            'success'                => $data['transaction']['success'] ?? false,
            'transaction_created_at' => $data['transaction']['created_at'] ?? now(),
            'is_3d_secure'           => $data['transaction']['is_3d_secure'] ?? false,
            'payment_method'         => $data['transaction']['source_data']['type'] ?? null,
            'card_type'              => $data['transaction']['source_data']['sub_type'] ?? null,
            'card_last4'             => $data['transaction']['source_data']['pan'] ?? null,
            'receipt_number'         => $data['transaction']['receipt'] ?? null,
            'customer_email'         => $data['intention']['intention_detail']['billing_data']['email'] ?? null,
            'customer_phone'         => $data['intention']['intention_detail']['billing_data']['phone_number'] ?? null,
            'items'                  => json_encode($data['intention']['intention_detail']['items'] ?? [])
        ];
    }


    protected function generateCourseInvoice(CourseEnrollment $enrollment, array $paymentData): string
    {
        $course = $enrollment->course;
        $user = $enrollment->user;
        $amount = $paymentData['transaction']['amount_cents'] / 100;

        $invoiceService = new GenerateInvoice();

        return $invoiceService
            ->setCustomer([
                'name' => $user->name,
                'address' => $user->address ?? 'N/A',
                'email' => $user->email,
                'phone' => $user->phone ?? 'N/A',
            ])
            ->setSeller([
                'name' => config('app.name'),
                'address' => config('app.address'),
                'email' => config('app.email'),
                'phone' => config('app.phone'),
            ])
            ->addItem(
                $course->title,
                $amount,
                1
            )
            ->setLogo(config('app.logo'))
            ->setInvoiceNumber($enrollment->id)
            ->saveToDisk('public', 'invoices')
            ->save();
    }


}
