<?php

namespace App\Http\Controllers\Api\V1\ClientSide;

use App\Http\Controllers\Controller;
use App\Http\Resources\GradeLecture\GradeLectureResource;
use App\Http\Resources\LectureQuestionClientSide\QuestionResource;
use App\Http\Traits\ApiResponse;
use App\Http\Traits\NotificationTrait;
use App\Models\Lecture;
use App\Models\LectureQuestion;
use App\Models\Question;
use App\Models\QuestionOption;
use App\Models\StudentAnswer;
use App\Models\StudentGrade;
use Atomescrochus\StringSimilarities\Compare;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class LectureQuestionController extends Controller
{
    use NotificationTrait, ApiResponse;
    public function show($id): JsonResponse
    {
        $lecture=Lecture::where('id',$id)->firstOrFail();
        $lectureQuestion=LectureQuestion::where('lecture_id',$lecture->id)->get();
        return $this->apiResponseShow( QuestionResource::collection($lectureQuestion->load('options','case')));
    }

    public function answer(Request $request)
    {
        $validated = $request->validate([
            'lecture_id' => 'required|exists:lectures,id',
            'answers' => 'required|array',
            'answers.*.question_id' => 'required|exists:lecture_questions,id',
            'answers.*.question_type' => 'required|integer|between:0,3',
            'answers.*.option_id' => 'nullable|required_if:answers.*.question_type,0,1|exists:question_options,id',
            'answers.*.answer_content' => 'nullable|required_if:answers.*.question_type,2|string'
        ]);

        $studentId = auth()->id();
        $lectureId = $validated['lecture_id'];

        if (StudentAnswer::where('lecture_id', $lectureId)
                    ->where('student_id', $studentId)
                    ->exists()) {
            return $this->apiResponseFailed('Answers for this lecture already submitted');
        }

        $totalQuestions = count($validated['answers']);
        $correctAnswers = 0;
        $createdAnswers = [];
        foreach ($validated['answers'] as $answer) {
            $isCorrect = null;

            if (in_array($answer['question_type'], [0, 1, 3]) && isset($answer['option_id'])) {
                $isCorrect = QuestionOption::find($answer['option_id'])->is_correct;
                 if ($isCorrect) {
                    $correctAnswers++;
                }
            }

            if ($answer['question_type'] == 2) {
                $question = LectureQuestion::findOrFail($answer['question_id']);

                $compare = new Compare();

                $studentAnswer = $this->normalizeAnswer($answer['answer_content']);
                $modelAnswer = $this->normalizeAnswer($question->answer_writing);

                $similarityScore = $compare->similarText($studentAnswer, $modelAnswer);

                // $normalizedScore = $similarityScore / 100;

                $isCorrect = $similarityScore >= 50;

                if ($isCorrect) {
                    $correctAnswers++;
                    $isCorrect=1;
                } else {
                    $isCorrect=0;
                }
            }


            $createdAnswer = StudentAnswer::create([
                'student_id' => $studentId,
                'lecture_id' => $lectureId,
                'question_id' => $answer['question_id'],
                'question_type' => $answer['question_type'],
                'answer_content' => $answer['answer_content'] ?? null,
                'option_id' => $answer['option_id'] ?? null,
                'is_correct' => $isCorrect
            ]);

            $createdAnswers[] = $createdAnswer;
        }

        $grade = ($totalQuestions > 0) ? ($correctAnswers / $totalQuestions) * 100 : 0;

        $studentGrade=StudentGrade::create([
            'student_id' => auth()->user()->id,
            'lecture_id' => $lectureId,
            'total_questions' => $totalQuestions,
            'correct_answers' => $correctAnswers,
            'grade' => $grade,
        ]);

        $this->sendPushNotificationClientSide('client_side', 'answer_lecture');

        return $this->apiResponseShow([
            'count' => count($createdAnswers),
            'answers' => $createdAnswers,
            'grade' => $studentGrade,
        ]);
    }

    public function gradeLecture($id){
        $lecture=Lecture::where('id',$id)->firstOrFail();
        $grades=StudentGrade::where('lecture_id',$lecture->id)->where('student_id',auth()->user()->id)->get();
        return $this->apiResponseShow( GradeLectureResource::collection($grades));
    }

    protected function normalizeAnswer(string $answer): string
    {
        $answer = mb_strtolower(trim($answer));
        $answer = preg_replace('/[^\p{L}\p{N}\s]/u', '', $answer);
        $answer = preg_replace('/\s+/', ' ', $answer);
        return $answer;
    }

}
