<?php

namespace App\Http\Controllers\Api\V1;

use App\Helpers\ValidationMessageHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\Category\UpdateCategoryRequest;
use App\Http\Requests\Category\CreateCategoryRequest;
use App\Http\Resources\Category\CategoryResource;
use App\Models\Category;
use Illuminate\Http\JsonResponse;

class CategoryController extends Controller
{
    public function index()
    {
        $categories = Category::useFilters()
            ->orderBy('order')
            ->dynamicPaginate();
        return CategoryResource::collection($categories);
    }

    public function store(CreateCategoryRequest $request): JsonResponse
    {
        $data = $request->validated();

        // if (!isset($data['order']) && isset($data['name'])) {
        //     $data['order'] = $this->getCategoryOrder($data['name']);
        // }

        $category = Category::create($data);
        return $this->apiResponseStored(new CategoryResource($category));
    }

    public function show(Category $category): JsonResponse
    {
        return $this->apiResponseShow(new CategoryResource($category));
    }

    public function update(UpdateCategoryRequest $request, Category $category): JsonResponse
    {
        $data = $request->validated();

        // if (isset($data['name']) && !isset($data['order'])) {
        //     $data['order'] = $this->getCategoryOrder($data['name']);
        // }

        $category->update($data);
        return $this->apiResponseUpdated(new CategoryResource($category));
    }

    public function destroy(Category $category): JsonResponse
    {
        if ($category->courses()->exists()) {
            return $this->apiResponse([], ValidationMessageHelper::cannotDelete('category', 'courses'), 422);
        }

        $category->delete();
        return $this->apiResponseDeleted();
    }

    private function getCategoryOrder(string $name): int
    {
        $orderMap = [
            'A0' => 1,
            'A0/A1' => 1,
            'A1' => 2,
            'A2' => 3,
            'B1' => 4,
            'B2' => 5,
            'C1' => 6,
            'C2' => 7,
        ];

        if (isset($orderMap[$name])) {
            return $orderMap[$name];
        }

        preg_match('/([A-C][0-2])/', $name, $matches);
        if (!empty($matches[1]) && isset($orderMap[$matches[1]])) {
            return $orderMap[$matches[1]];
        }

        return 99;
    }
}
